<?php

namespace Drupal\field_inheritance\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\ConfigTarget;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\KeyValueStore\KeyValueExpirableFactoryInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * Provides a form for configuring field inheritance settings.
 *
 * @ingroup field_inheritance
 */
class FieldInheritanceSettingsForm extends ConfigFormBase {

  use AutowireTrait;

  /**
   * Returns a unique string identifying the form.
   *
   * @return string
   *   The unique string identifying the form.
   */
  public function getFormId() {
    return 'field_inheritance_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['field_inheritance.config'];
  }

  /**
   * Construct a FieldInheritanceSettingsForm.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typedConfigManager
   *   The typed config manager.
   * @param \Drupal\Core\Entity\EntityTypeManager $entityTypeManager
   *   The entity type manager service.
   * @param \Drupal\Core\KeyValueStore\KeyValueExpirableFactoryInterface $keyValueExpirable
   *   The keyvalue expirable factory.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    TypedConfigManagerInterface $typedConfigManager,
    protected EntityTypeManagerInterface $entityTypeManager,
    #[Autowire(service: 'keyvalue.expirable')]
    protected KeyValueExpirableFactoryInterface $keyValueExpirable,
  ) {
    parent::__construct($config_factory, $typedConfigManager);
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $entity_types = $this->entityTypeManager->getDefinitions();
    $entity_types = array_filter($entity_types, function ($type) {
      return $type->entityClassImplements(FieldableEntityInterface::CLASS);
    });
    $entity_types = array_map(fn (EntityTypeInterface $type) => $type->getLabel(), $entity_types);

    $form['included_entities'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Included Entity Types'),
      '#description' => $this->t('Select the entity types that should be able to inherit data'),
      '#options' => $entity_types,
      '#config_target' => new ConfigTarget('field_inheritance.config', 'included_entities', toConfig: static::class . '::filterAndSortAllowedTypes'),

    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $config = $this->config('field_inheritance.config');
    $original_entity_types = is_array($config->getOriginal('included_entities')) ?
      $config->getOriginal('included_entities') : [];
    $entity_types = is_array($config->get('included_entities')) ?
      $config->get('included_entities') : [];

    $to_remove = array_diff($original_entity_types, $entity_types);
    // Not removing anything, we can continue.
    if (empty($to_remove)) {
      parent::submitForm($form, $form_state);
      return;
    }

    $user_id = $this->currentUser()->id();
    // Store the values for 6 hours. This expiration time is also used in
    // the form cache.
    $this->keyValueExpirable->get('field_inheritance_to_remove')->setWithExpire($user_id, $entity_types, 6 * 60 * 60);

    // Redirect to the confirm form.
    $form_state->setRedirect('field_inheritance.settings_confirm');
  }

  /**
   * Transformation callback for the book_allowed_types config value.
   *
   * @param array $allowed_types
   *   The config value to transform.
   *
   * @return array
   *   The transformed value.
   */
  public static function filterAndSortAllowedTypes(array $allowed_types): array {
    $allowed_types = array_filter($allowed_types);
    // We need to save the allowed types in an array ordered by machine_name.
    sort($allowed_types);
    return $allowed_types;
  }

}
