<?php

declare(strict_types=1);

namespace Drupal\field_inheritance\Hook;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\field\Entity\FieldConfig;
use Drupal\field_inheritance\Entity\FieldInheritanceInterface;
use Drupal\field_inheritance\EntityReferenceFieldInheritanceFactory;
use Drupal\field_inheritance\FieldInheritanceFactory;
use Drupal\field_inheritance\FieldStorageDefinition;

/**
 * Hook implementations for field_inheritance.
 */
class FieldInheritanceHooks {

  use StringTranslationTrait;

  /**
   * Constructs a new FieldInheritanceHooks.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entityFieldManager
   *   The entity field manager.
   * @param \Drupal\Core\Session\AccountInterface $currentUser
   *   The current user.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected ModuleHandlerInterface $moduleHandler,
    protected EntityFieldManagerInterface $entityFieldManager,
    protected AccountInterface $currentUser,
    protected ConfigFactoryInterface $configFactory,
  ) {}

  /**
   * Implements hook_help().
   */
  #[Hook('help')]
  public function help(string $route_name, RouteMatchInterface $route_match): ?string {
    switch ($route_name) {
      // Main module help for the field_inheritance module.
      case 'help.page.field_inheritance':
        $output = '';
        $output .= '<h3>' . t('About') . '</h3>';
        $output .= '<p>' . t('The field inheritance module can be considered to be a field-level entity reference alternative. It allows site administrators to inherit any field from any entity into any other entity.') . '</p>';
        $output .= '<p>' . t('For more information, see the <a href=":docs">online documentation for the Field Inheritance module</a>.', [
          ':docs' => 'https://www.drupal.org/project/field_inheritance',
        ]) . '</p>';
        return $output;
    }

    return NULL;
  }

  /**
   * Implements hook_entity_base_field_info().
   */
  #[Hook('entity_base_field_info')]
  public function entityBaseFieldInfo(EntityTypeInterface $entity_type): array {
    $fields = [];
    $entity_types = $this->configFactory->get('field_inheritance.config')->get('included_entities');
    $entity_types = is_array($entity_types) ? $entity_types : [];

    if (in_array($entity_type->id(), $entity_types)) {
      $fields['field_inheritance'] = BaseFieldDefinition::create('map')
        ->setLabel($this->t('Field Inheritance'))
        ->setDescription($this->t('A serialized array of order options.'))
        ->setRevisionable($entity_type->isRevisionable())
        ->setDisplayConfigurable('form', TRUE);
    }

    return $fields;
  }

  /**
   * Implements hook_entity_bundle_field_info_alter().
   */
  #[Hook('entity_bundle_field_info_alter')]
  public function entityBundleFieldInfoAlter(&$fields, EntityTypeInterface $entity_type, $bundle): void {
    $field_inheritance_storage = $this->entityTypeManager
      ->getStorage('field_inheritance');
    // Grab inherited fields just for the current entity type and bundle.
    $inherited_field_ids = $field_inheritance_storage
      ->getQuery()
      ->accessCheck(FALSE)
      ->condition('destinationEntityType', $entity_type->id())
      ->condition('destinationEntityBundle', $bundle)
      ->execute();

    if (!empty($inherited_field_ids)) {
      $inherited_fields = $field_inheritance_storage->loadMultiple($inherited_field_ids);
      if (!empty($inherited_fields)) {
        $x = 0;
        foreach ($inherited_fields as $field) {
          // We are only interested in adding computed fields to the destination
          // entity type.
          if ($field->destinationEntityType() !== $entity_type->id()) {
            continue;
          }

          // We are only interested in adding computed fields to the destination
          // entity bundle.
          if ($field->destinationEntityBundle() !== $bundle) {
            continue;
          }

          // Configure the settings array with all the values from the config
          // entity.
          $settings = [
            'id' => $field->idWithoutTypeAndBundle(),
            'source entity type' => $field->sourceEntityType(),
            'source entity bundle' => $field->sourceEntityBundle(),
            'source identifier' => $field->sourceIdentifier(),
            'source field' => $field->sourceField(),
            'method' => $field->type(),
            'plugin' => $field->plugin(),
          ];

          // Only set the destination field if one was configured, which will be
          // for inheritance strategies other than inherit.
          if ($field->destinationField()) {
            $settings['destination field'] = $field->destinationField();
          }

          // Grab the configuration of the source field.
          $source_field = $this->getSourceFieldDefinition($field);
          if (!$source_field) {
            continue;
          }
          $settings = array_merge($settings, $source_field->getSettings());
          $type = $source_field->getType();
          $cardinality = $this->getComputedFieldCardinality($field);

          // Set the factory class used to inherit the data from the source.
          $class = $field->plugin() === 'entity_reference_inheritance' ?
            EntityReferenceFieldInheritanceFactory::class :
            FieldInheritanceFactory::class;

          // Allow developers to override the class to use for a field.
          $this->moduleHandler->alter('field_inheritance_inheritance_class', $class, $field);

          // Add a computed field for the inherited field to the appropriate
          // destination entity type and bundle.
          $fields[$field->idWithoutTypeAndBundle()] = FieldStorageDefinition::create($type)
            ->setLabel(t('@label', ['@label' => $field->label()]))
            ->setName($field->idWithoutTypeAndBundle())
            ->setDescription(t('The inherited field: @field', ['@field' => $field->label()]))
            ->setComputed(TRUE)
            ->setCardinality($cardinality)
            ->setClass($class)
            ->setSettings($settings)
            ->setTargetEntityTypeId($field->destinationEntityType())
            ->setTargetBundle($field->destinationEntityBundle())
            ->setTranslatable(FALSE)
            ->setRevisionable(FALSE)
            ->setReadOnly(TRUE)
            ->setDisplayConfigurable('view', TRUE)
            ->setDisplayOptions('view', [
              'label' => 'visible',
              'weight' => (50 + $x),
            ])
            ->setProvider('field_inheritance');
          $x++;
        }
      }
    }
  }

  /**
   * Gets the source field definition for a given field inheritance.
   *
   * @param \Drupal\field_inheritance\Entity\FieldInheritanceInterface $field_inheritance
   *   The field inheritance.
   *
   * @return \Drupal\Core\Field\FieldDefinitionInterface|null
   *   The source field definition if available. NULL otherwise.
   */
  protected function getSourceFieldDefinition(FieldInheritanceInterface $field_inheritance): ?FieldDefinitionInterface {
    if (!$field_inheritance->sourceField()) {
      return NULL;
    }

    $field = FieldConfig::loadByName($field_inheritance->sourceEntityType(), $field_inheritance->sourceEntityBundle(), $field_inheritance->sourceField());
    if (!empty($field)) {
      return $field;
    }

    $fields = $this->entityFieldManager->getBaseFieldDefinitions($field_inheritance->sourceEntityType());
    return $fields[$field_inheritance->sourceField()] ?? NULL;
  }

  /**
   * Gets the destination field definition for a given field inheritance.
   *
   * @param \Drupal\field_inheritance\Entity\FieldInheritanceInterface $field_inheritance
   *   The field inheritance.
   *
   * @return \Drupal\Core\Field\FieldDefinitionInterface|null
   *   The destination field definition if available. NULL otherwise.
   */
  protected function getDestinationFieldDefinition(FieldInheritanceInterface $field_inheritance): ?FieldDefinitionInterface {
    if (!$field_inheritance->destinationField()) {
      return NULL;
    }

    $field = FieldConfig::loadByName($field_inheritance->destinationEntityType(), $field_inheritance->destinationEntityBundle(), $field_inheritance->destinationField());
    if (!empty($field)) {
      return $field;
    }

    $fields = $this->entityFieldManager->getBaseFieldDefinitions($field_inheritance->destinationEntityType());
    return $fields[$field_inheritance->destinationField()] ?? NULL;
  }

  /**
   * Gets the computed cardinality for a given field inheritance.
   *
   * @param \Drupal\field_inheritance\Entity\FieldInheritanceInterface $field_inheritance
   *   The field inheritance.
   *
   * @return int
   *   The computed cardinality.
   */
  protected function getComputedFieldCardinality(FieldInheritanceInterface $field_inheritance): int {
    $source_field = $this->getSourceFieldDefinition($field_inheritance);
    $source_cardinality = $source_field?->getFieldStorageDefinition()->getCardinality() ?? 1;
    $destination_field = $this->getDestinationFieldDefinition($field_inheritance);
    $destination_cardinality = $destination_field?->getFieldStorageDefinition()->getCardinality() ?? 1;

    $unlimited = in_array(FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED, [
      $source_cardinality,
      $destination_cardinality,
    ]);

    return match ($field_inheritance->type()) {
      'prepend', 'append' => $unlimited ?
        FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED :
        $source_cardinality + $destination_cardinality,
      'fallback' => $unlimited ?
        FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED :
        max($source_cardinality, $destination_cardinality),
      default => $source_cardinality,
    };
  }

}
