<?php

declare(strict_types=1);

namespace Drupal\field_inheritance\Plugin\Field\FieldWidget;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\Attribute\FieldWidget;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\field_inheritance\Entity\FieldInheritanceInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines the 'field_inheritance_default' field widget.
 */
#[FieldWidget(
  id: 'field_inheritance_default',
  label: new TranslatableMarkup('Field Inheritance'),
  field_types: ['map'],
)]
final class FieldInheritanceDefaultWidget extends WidgetBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs the plugin instance.
   */
  public function __construct(
    $plugin_id,
    $plugin_definition,
    FieldDefinitionInterface $field_definition,
    array $settings,
    array $third_party_settings,
    private readonly EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $third_party_settings);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['third_party_settings'],
      $container->get('entity_type.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings(): array {
    return [
      'elements' => 'both',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state): array {
    $element['elements'] = [
      '#type' => 'select',
      '#title' => t('Elements to show'),
      '#description' => $this->t('Choose which fields to display in the widget. Fields Mappings will override specific Entities Mappings.'),
      '#options' => $this->elementOptions(),
      '#default_value' => $this->getSetting('elements'),
      '#required' => TRUE,
    ];
    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary(): array {
    $options = $this->elementOptions();
    $elements = $this->getSetting('elements');
    return [
      $this->t('Elements included: @elements', ['@elements' => $options[$elements] ?? $elements]),
    ];
  }

  /**
   * Returns the list of element options for this widget.
   *
   * @return array
   *   The element options.
   */
  protected function elementOptions(): array {
    return [
      'entities' => $this->t('Entity Mappings'),
      'fields' => $this->t('Field Mappings'),
      'both' => $this->t('Both'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state): array {
    $field_inheritance_storage = $this->entityTypeManager
      ->getStorage('field_inheritance');
    $inherited_field_ids = $field_inheritance_storage
      ->getQuery()
      ->accessCheck(FALSE)
      ->condition('destinationEntityType', $items->getEntity()->getEntityTypeId())
      ->condition('destinationEntityBundle', $items->getEntity()->bundle())
      ->execute();

    if (empty($inherited_field_ids)) {
      return [];
    }
    $element_path = $items->getFieldDefinition()->getName() . '[' . $delta . ']';
    $element['field_inheritance'] = [
      '#type' => 'details',
      '#open' => TRUE,
      '#title' => t('Field Inheritance'),
    ];

    $element['field_inheritance']['enabled'] = [
      '#type' => 'checkbox',
      '#title' => t('Enable inheritance for this entity?'),
      '#description' => t('Enabling inheritance will allow data to be pulled in from a source entity into this entity.'),
      '#default_value' => $items[$delta]->enabled ?? NULL,
      '#weight' => -10,
    ];

    $element['field_inheritance']['fields'] = [
      '#type' => 'details',
      '#open' => FALSE,
      '#title' => t('Field Mappings'),
      '#description' => $this->t('Field Mappings define a specific entity for each Field Inheritance.'),
      '#weight' => 10,
      '#states' => [
        'visible' => [
          'input[name="' . $element_path . '[field_inheritance][enabled]"]' => ['checked' => TRUE],
        ],
      ],
      '#access' => $this->getSetting('elements') !== 'entities',
    ];

    $inherited_fields = $field_inheritance_storage->loadMultiple($inherited_field_ids);
    $source_bundles = [];

    foreach ($inherited_fields as $field) {
      assert($field instanceof FieldInheritanceInterface);
      $source_bundles[$field->sourceIdentifier()][] = $field->label();
      $element['field_inheritance']['fields'][$field->idWithoutTypeAndBundle()] = [
        '#type' => 'details',
        '#open' => TRUE,
        '#title' => t('Entity Mapping for @label [@entity: @bundle - @field]', [
          '@label' => $field->label(),
          '@entity' => $field->sourceEntityType(),
          '@bundle' => $field->sourceEntityBundle(),
          '@field' => $field->sourceField(),
        ]),
        'skip' => [
          '#type' => 'checkbox',
          '#title' => t('Do not inherit this field'),
          '#description' => t('Checking this box will prevent data from being inherited for this field on this entity.'),
          '#default_value' => $items[$delta]->fields[$field->idWithoutTypeAndBundle()]['skip'] ?? FALSE,
        ],
        'entity' => [
          '#type' => 'entity_autocomplete',
          '#title' => t('Source Entity'),
          '#description' => t('Enter the name of the %bundle from which to inherit data.', [
            '%bundle' => $field->sourceEntityBundle(),
          ]),
          '#target_type' => $field->sourceEntityType(),
          '#selection_settings' => ['target_bundles' => [$field->sourceEntityBundle()]],
          '#default_value' => !empty($items[$delta]->fields[$field->idWithoutTypeAndBundle()]['entity']) ?
          $this->entityTypeManager->getStorage($field->sourceEntityType())->load($items[$delta]->fields[$field->idWithoutTypeAndBundle()]['entity']) :
          FALSE,
          '#states' => [
            'invisible' => [
              'input[name="' . $element_path . '[field_inheritance][fields][' . $field->idWithoutTypeAndBundle() . '][skip]"]' => ['checked' => TRUE],
            ],
          ],
        ],
      ];
    }

    $element['field_inheritance']['entities'] = [
      '#type' => 'details',
      '#open' => FALSE,
      '#title' => t('Entity Mappings'),
      '#description' => $this->t('Entity Mappings allow to define a common entity for each the entity/bundle Field Inheritance combinations.'),
      '#weight' => 5,
      '#states' => [
        'visible' => [
          'input[name="' . $element_path . '[field_inheritance][enabled]"]' => ['checked' => TRUE],
        ],
      ],
      '#access' => $this->getSetting('elements') !== 'fields',
    ];

    foreach ($source_bundles as $source_bundle => $source_fields) {
      $element['field_inheritance']['entities'][$source_bundle] = [
        '#type' => 'details',
        '#open' => TRUE,
        '#title' => t('Entity Mapping for [@entity: @bundle] (@label)', [
          '@label' => implode(', ', $source_fields),
          '@entity' => $field->sourceEntityType(),
          '@bundle' => $field->sourceEntityBundle(),
          '@field' => $field->sourceField(),
        ]),
        'entity' => [
          '#type' => 'entity_autocomplete',
          '#title' => t('Source Entity'),
          '#description' => t('Enter the name of the %bundle from which to inherit data.', [
            '%bundle' => $field->sourceEntityBundle(),
          ]),
          '#target_type' => $field->sourceEntityType(),
          '#selection_settings' => ['target_bundles' => [$field->sourceEntityBundle()]],
          '#default_value' => !empty($items[$delta]->entities[$source_bundle]['entity']) ?
          $this->entityTypeManager->getStorage($field->sourceEntityType())->load($items[$delta]->entities[$source_bundle]['entity']) :
          FALSE,
        ],
      ];
    }

    if ($this->getSetting('elements') === 'both') {
      $element['field_inheritance']['fields']['#description'] = $this->t('Field Mappings define a specific entity for each Field Inheritance. Field Mappings override Entity Mappings.');
    }
    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function massageFormValues(array $values, array $form, FormStateInterface $form_state) {
    return array_map(function ($item) {
      if (isset($item['field_inheritance']) && is_array($item['field_inheritance'])) {
        // Merge and override keys from field_inheritance.
        $item = array_merge($item['field_inheritance'], $item);
        unset($item['field_inheritance']);
      }
      return $item;
    }, $values);
  }

}
