<?php

namespace Drupal\field_inheritance\Plugin\FieldInheritance;

use Drupal\Component\Plugin\PluginBase;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\field_inheritance\FieldInheritancePluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Abstract class FieldInheritancePluginBase.
 */
abstract class FieldInheritancePluginBase extends PluginBase implements FieldInheritancePluginInterface, ContainerFactoryPluginInterface {

  /**
   * The field inheritance id.
   *
   * @var string
   */
  protected string $fieldInheritanceId;

  /**
   * The entity.
   *
   * @var \Drupal\Core\Entity\EntityInterface
   */
  protected EntityInterface $entity;

  /**
   * The method used to inherit.
   *
   * @var string
   */
  protected string $method;

  /**
   * The source entity type used to inherit.
   *
   * @var string
   */
  protected string $sourceEntityType;

  /**
   * The source entity bundle used to inherit.
   *
   * @var string
   */
  protected string $sourceEntityBundle;

  /**
   * The source identifier used to inherit.
   *
   * @var string
   */
  protected string $sourceIdentifier;

  /**
   * The source field used to inherit.
   *
   * @var string
   */
  protected string $sourceField;

  /**
   * The entity field used to inherit.
   *
   * @var string|null
   */
  protected string|null $destinationField;

  /**
   * The current language code.
   *
   * @var string
   */
  protected string $langCode;

  /**
   * Constructs a FieldInheritancePluginBase object.
   *
   * @param array $configuration
   *   The plugin configuration.
   * @param string $plugin_id
   *   The plugin ID.
   * @param mixed $plugin_definition
   *   The plugin definition.
   * @param \Drupal\Core\Language\LanguageManagerInterface $languageManager
   *   The language manager service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The key value store.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected LanguageManagerInterface $languageManager,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected ModuleHandlerInterface $moduleHandler,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->fieldInheritanceId = $configuration['id'];
    $this->entity = $configuration['entity'];
    $this->method = $configuration['method'];
    $this->sourceEntityType = $configuration['source entity type'];
    $this->sourceEntityBundle = $configuration['source entity bundle'];
    $this->sourceIdentifier = $configuration['source identifier'];
    $this->sourceField = $configuration['source field'];
    $this->destinationField = $configuration['destination field'] ?? NULL;
    $this->langCode = $this->languageManager->getCurrentLanguage()->getId();
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('language_manager'),
      $container->get('entity_type.manager'),
      $container->get('module_handler'),
    );
  }

  /**
   * Get the configuration method.
   *
   * @return string|null
   *   The configuration method.
   */
  public function getMethod(): string|null {
    return $this->method;
  }

  /**
   * Get the configuration source entity type.
   *
   * @return string|null
   *   The configuration source entity type.
   */
  public function getSourceEntityType(): string|null {
    return $this->sourceEntityType;
  }

  /**
   * Get the configuration source entity bundle.
   *
   * @return string|null
   *   The configuration source entity bundle.
   */
  public function getSourceEntityBundle(): string|null {
    return $this->sourceEntityBundle;
  }

  /**
   * Get the inheritance source identifier.
   *
   * @return string
   *   The inheritance source identifier.
   */
  public function getSourceIdentifier(): string {
    return $this->sourceIdentifier;
  }

  /**
   * Get the configuration source field.
   *
   * @return string|null
   *   The configuration source field.
   */
  public function getSourceField(): string|null {
    return $this->sourceField;
  }

  /**
   * Get the configuration destination entity type.
   *
   * @return string|null
   *   The configuration destination entity type.
   */
  public function getDestinationEntityType(): string|null {
    return $this->entity?->getEntityTypeId();
  }

  /**
   * Get the configuration destination entity bundle.
   *
   * @return string|null
   *   The configuration destination entity bundle.
   */
  public function getDestinationEntityBundle(): string|null {
    return $this->entity?->bundle();
  }

  /**
   * Get the configuration destination field.
   *
   * @return string|null
   *   The configuration destination field.
   */
  public function getDestinationField(): string|null {
    return $this?->destinationField;
  }

  /**
   * {@inheritdoc}
   */
  public function computeValue(): array {
    $this->validateArguments();
    $method = $this->getMethod();

    $value = [];
    switch ($method) {
      case 'inherit':
        $value = $this->inheritData();
        break;

      case 'prepend':
        $value = $this->prependData();
        break;

      case 'append':
        $value = $this->appendData();
        break;

      case 'fallback':
        $value = $this->fallbackData();
        break;
    }

    $context = [
      'source_field' => $this->getSourceField(),
      'source_entity' => $this->getSourceEntity(),
      'destination_field' => $this->getDestinationField(),
      'destination_entity' => $this->getDestinationEntity(),
      'method' => $this->getMethod(),
    ];
    $this->moduleHandler->alter('field_inheritance_compute_value', $value, $context);

    return $value;
  }

  /**
   * Retrieve inherited data.
   *
   * @return array
   *   The inherited data.
   */
  protected function inheritData(): array {
    $source_entity = $this->getSourceEntity();
    if ($source_entity === FALSE
      || $source_entity->{$this->getSourceField()}->isEmpty()
    ) {
      return [];
    }

    return $source_entity->{$this->getSourceField()}->getValue();
  }

  /**
   * Retrieve prepended data.
   *
   * @return array
   *   The prepended data.
   */
  protected function prependData(): array {
    $source_entity = $this->getSourceEntity();
    $destination_entity = $this->getDestinationEntity();
    $values = [];

    if ($source_entity === FALSE) {
      return $values;
    }

    if (!$destination_entity->{$this->getDestinationField()}->isEmpty()) {
      $values = array_merge($values, $destination_entity->{$this->getDestinationField()}->getValue());
    }
    if (!$source_entity->{$this->getSourceField()}->isEmpty()) {
      $values = array_merge($values, $source_entity->{$this->getSourceField()}->getValue());
    }
    return $values;
  }

  /**
   * Retrieve appended data.
   *
   * @return array
   *   The appended data.
   */
  protected function appendData(): array {
    $source_entity = $this->getSourceEntity();
    $destination_entity = $this->getDestinationEntity();
    $values = [];

    if ($source_entity === FALSE) {
      return $values;
    }

    if (!$source_entity->{$this->getSourceField()}->isEmpty()) {
      $values = array_merge($values, $source_entity->{$this->getSourceField()}->getValue());
    }
    if (!$destination_entity->{$this->getDestinationField()}->isEmpty()) {
      $values = array_merge($values, $destination_entity->{$this->getDestinationField()}->getValue());
    }
    return $values;
  }

  /**
   * Retrieve fallback data.
   *
   * @return array
   *   The fallback data.
   */
  protected function fallbackData(): array {
    $source_entity = $this->getSourceEntity();
    $destination_entity = $this->getDestinationEntity();
    $values = [];

    if ($source_entity === FALSE) {
      return $values;
    }

    if (!$destination_entity->{$this->getDestinationField()}->isEmpty()) {
      $values = $destination_entity->{$this->getDestinationField()}->getValue();
    }
    elseif (!$source_entity->{$this->getSourceField()}->isEmpty()) {
      $values = $source_entity->{$this->getSourceField()}->getValue();
    }
    return $values;
  }

  /**
   * Validate the configuration arguments of the plugin.
   */
  protected function validateArguments(): bool {
    if (empty($this->getMethod())) {
      throw new \InvalidArgumentException("The definition's 'method' key must be set to inherit data.");
    }

    if (empty($this->getSourceField())) {
      throw new \InvalidArgumentException("The definition's 'source field' key must be set to inherit data.");
    }

    $method = $this->getMethod();
    $destination_field_methods = [
      'prepend',
      'append',
      'fallback',
    ];

    if (array_search($method, $destination_field_methods)) {
      if (empty($this->getDestinationField())) {
        throw new \InvalidArgumentException("The definition's 'destination field' key must be set to prepend, append, or fallback to series data.");
      }
    }

    return TRUE;
  }

  /**
   * Get the translated source entity.
   *
   * @return \Drupal\Core\Entity\EntityInterface|false
   *   The translated source entity, or FALSE.
   */
  protected function getSourceEntity(): EntityInterface|false {
    $entity = $this->entity;
    if (empty($entity)) {
      return FALSE;
    }
    if (!$entity->hasField('field_inheritance') || $entity->get('field_inheritance')->isEmpty()) {
      return FALSE;
    }

    $values = $entity->get('field_inheritance')->first()->toArray();

    if (empty($values['enabled'])) {
      return FALSE;
    }

    $source_id = $values['fields'][$this->fieldInheritanceId]['entity'] ?? ($values['entities'][$this->sourceIdentifier]['entity'] ?? NULL);

    if (!empty($source_id) && empty($values['fields'][$this->fieldInheritanceId]['skip'])) {
      if ($source = $this->entityTypeManager->getStorage($this->sourceEntityType)->load($source_id)) {
        $context['data'] = $source;
        $context += [
          'operation' => 'entity_view',
          'langcode' => $this->langCode,
        ];
        $candidates = $this->languageManager->getFallbackCandidates($context);
        foreach ($candidates as $candidate) {
          if ($source->hasTranslation($candidate)) {
            return $source->getTranslation($candidate);
          }
        }
      }
    }

    return FALSE;
  }

  /**
   * Get the translated destination entity.
   *
   * @return \Drupal\Core\Entity\EntityInterface
   *   The translated destination entity.
   */
  protected function getDestinationEntity(): EntityInterface {
    $context['data'] = $this->entity;
    $context += [
      'operation' => 'entity_view',
      'langcode' => $this->langCode,
    ];
    $candidates = $this->languageManager->getFallbackCandidates($context);
    foreach ($candidates as $candidate) {
      if ($this->entity->hasTranslation($candidate)) {
        return $this->entity->getTranslation($candidate);
      }
    }
    return $this->entity;
  }

}
