<?php

declare(strict_types=1);

namespace Drupal\Tests\field_inheritance\Kernel;

use Drupal\entity_test\Entity\EntityTest;

/**
 * Tests the field_inheritance field behavior.
 *
 * @group field_inheritance
 */
class FieldInheritanceFieldTest extends FieldInheritanceTestBase {

  /**
   * The parent entity.
   *
   * @var \Drupal\entity_test\Entity\EntityTest
   */
  protected EntityTest $parentNew;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create parent entity with field values.
    $this->parentNew = EntityTest::create([
      'name' => 'Parent New entity',
      'field_one' => [
        ['value' => 'Parent New note 1'],
        ['value' => 'Parent New note 2'],
      ],
      'field_two' => [
        ['value' => 'Parent New note 3'],
      ],
    ]);
    $this->parentNew->save();
  }

  /**
   * Tests the field behavior.
   */
  public function testInheritanceField(): void {
    // By default the inheritance is set at entity level, so everything should
    // be taken from entity 1.
    $this->assertEquals($this->child->inheritance_one->getValue(), $this->parent->field_one->getValue());
    $this->assertEquals($this->child->inheritance_two->getValue(), $this->parent->field_two->getValue());

    // Override one of them at field level.
    $this->child->field_inheritance = [
      'enabled' => 1,
      'entities' => [
        'entity_test:entity_test' => ['entity' => $this->parent->id()],
      ],
      'fields' => [
        $this->inheritanceOne->idWithoutTypeAndBundle() => [
          'entity' => $this->parentNew->id(),
        ],
      ],
    ];

    $this->child->save();
    $this->child = \Drupal::entityTypeManager()->getStorage('entity_test')->loadUnchanged($this->child->id());
    $this->assertEquals($this->child->inheritance_one->getValue(), $this->parentNew->field_one->getValue());
    $this->assertEquals($this->child->inheritance_two->getValue(), $this->parent->field_two->getValue());

    // Skip one of them at field level.
    $this->child->field_inheritance = [
      'enabled' => 1,
      'entities' => [
        'entity_test:entity_test' => ['entity' => $this->parent->id()],
      ],
      'fields' => [
        $this->inheritanceOne->idWithoutTypeAndBundle() => [
          'entity' => $this->parentNew->id(),
          'skip' => 1,
        ],
      ],
    ];

    $this->child->save();
    $this->child = \Drupal::entityTypeManager()->getStorage('entity_test')->loadUnchanged($this->child->id());
    $this->assertEquals($this->child->inheritance_one->getValue(), []);
    $this->assertEquals($this->child->inheritance_two->getValue(), $this->parent->field_two->getValue());

    // Use only field level definitions.
    $this->child->field_inheritance = [
      'enabled' => 1,
      'entities' => [
        'entity_test:entity_test' => ['entity' => $this->parent->id()],
      ],
      'fields' => [
        $this->inheritanceOne->idWithoutTypeAndBundle() => [
          'entity' => $this->parentNew->id(),
        ],
        $this->inheritanceTwo->idWithoutTypeAndBundle() => [
          'entity' => $this->parent->id(),
        ],
      ],
    ];

    $this->child->save();
    $this->child = \Drupal::entityTypeManager()->getStorage('entity_test')->loadUnchanged($this->child->id());
    $this->assertEquals($this->child->inheritance_one->getValue(), $this->parentNew->field_one->getValue());
    $this->assertEquals($this->child->inheritance_two->getValue(), $this->parent->field_two->getValue());

    // Disable field_inheritance globally.
    $this->child->field_inheritance = [
      'enabled' => 0,
      'entities' => [
        'entity_test:entity_test' => ['entity' => $this->parent->id()],
      ],
      'fields' => [
        $this->inheritanceOne->idWithoutTypeAndBundle() => [
          'skip' => 1,
        ],
      ],
    ];

    $this->child->save();
    $this->child = \Drupal::entityTypeManager()->getStorage('entity_test')->loadUnchanged($this->child->id());
    $this->assertEquals($this->child->inheritance_one->getValue(), []);
    $this->assertEquals($this->child->inheritance_two->getValue(), []);
  }

}
