<?php

declare(strict_types=1);

namespace Drupal\Tests\field_inheritance\Kernel;

use Drupal\entity_test\Entity\EntityTest;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\field_inheritance\Entity\FieldInheritance;
use Drupal\KernelTests\KernelTestBase;

/**
 * Base class for Field Inheritance tests.
 */
abstract class FieldInheritanceTestBase extends KernelTestBase {

  /**
   * The 1st Field Inheritance.
   *
   * @var \Drupal\field_inheritance\Entity\FieldInheritance
   */
  protected FieldInheritance $inheritanceOne;

  /**
   * The 2nd Field Inheritance.
   *
   * @var \Drupal\field_inheritance\Entity\FieldInheritance
   */
  protected FieldInheritance $inheritanceTwo;

  /**
   * The parent entity.
   *
   * @var \Drupal\entity_test\Entity\EntityTest
   */
  protected EntityTest $parent;

  /**
   * The child entity.
   *
   * @var \Drupal\entity_test\Entity\EntityTest
   */
  protected EntityTest $child;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'field',
    'text',
    'field_inheritance',
    'entity_test',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('entity_test');
    $this->installSchema('user', ['users_data']);
    \Drupal::configFactory()->getEditable('field_inheritance.config')
      ->set('included_entities', ['entity_test'])
      ->save();

    // Create the field storage and config.
    FieldStorageConfig::create([
      'field_name' => 'field_one',
      'entity_type' => 'entity_test',
      'type' => 'string',
      'cardinality' => -1,
    ])->save();

    FieldConfig::create([
      'field_name' => 'field_one',
      'entity_type' => 'entity_test',
      'bundle' => 'entity_test',
      'label' => 'Field One',
    ])->save();

    // Create the field storage and config.
    FieldStorageConfig::create([
      'field_name' => 'field_two',
      'entity_type' => 'entity_test',
      'type' => 'string',
      'cardinality' => 3,
    ])->save();

    FieldConfig::create([
      'field_name' => 'field_two',
      'entity_type' => 'entity_test',
      'bundle' => 'entity_test',
      'label' => 'Field Two',
    ])->save();

    $this->inheritanceOne = FieldInheritance::create([
      'id' => 'entity_test_entity_test_inheritance_one',
    ]);
    $this->inheritanceOne->setType('inherit');
    $this->inheritanceOne->setSourceEntityType('entity_test');
    $this->inheritanceOne->setSourceEntityBundle('entity_test');
    $this->inheritanceOne->setSourceField('field_one');
    $this->inheritanceOne->setDestinationEntityType('entity_test');
    $this->inheritanceOne->setDestinationEntityBundle('entity_test');
    $this->inheritanceOne->setDestinationField('field_one');
    $this->inheritanceOne->setPlugin('default_inheritance');
    $this->inheritanceOne->save();

    $this->inheritanceTwo = FieldInheritance::create([
      'id' => 'entity_test_entity_test_inheritance_two',
    ]);
    $this->inheritanceTwo->setType('inherit');
    $this->inheritanceTwo->setSourceEntityType('entity_test');
    $this->inheritanceTwo->setSourceEntityBundle('entity_test');
    $this->inheritanceTwo->setSourceField('field_two');
    $this->inheritanceTwo->setDestinationEntityType('entity_test');
    $this->inheritanceTwo->setDestinationEntityBundle('entity_test');
    $this->inheritanceTwo->setDestinationField('field_two');
    $this->inheritanceTwo->setPlugin('default_inheritance');
    $this->inheritanceTwo->save();

    // Create parent entity with field values.
    $this->parent = EntityTest::create([
      'name' => 'Parent entity',
      'field_one' => [
        ['value' => 'Parent note 1'],
        ['value' => 'Parent note 2'],
      ],
      'field_two' => [
        ['value' => 'Parent note 3'],
      ],
    ]);
    $this->parent->save();

    // Create child entity with field values.
    $this->child = EntityTest::create([
      'name' => 'Child entity',
      'field_one' => [
        ['value' => 'Child note 1'],
        ['value' => 'Child note 2'],
      ],
      'field_two' => [
        ['value' => 'Child note 3'],
      ],
      'field_inheritance' => [
        'enabled' => 1,
        'entities' => [
          'entity_test:entity_test' => ['entity' => $this->parent->id()],
        ],
      ],
    ]);
    $this->child->save();
  }

}
