<?php

declare(strict_types=1);

namespace Drupal\Tests\field_inheritance\Kernel;

use Drupal\entity_test\Entity\EntityTest;

/**
 * Tests the different inheritance types behavior.
 *
 * @group field_inheritance
 */
class FieldInheritanceTypesTest extends FieldInheritanceTestBase {

  /**
   * Tests the 'inherit' inheritance type.
   */
  public function testInheritInheritanceType(): void {
    $this->assertEquals($this->child->inheritance_one->getValue(), $this->parent->field_one->getValue());
    $this->assertEquals($this->child->inheritance_two->getValue(), $this->parent->field_two->getValue());
    $fields = \Drupal::service('entity_field.manager')->getFieldDefinitions('entity_test', 'entity_test');
    $this->assertEquals(-1, $fields['inheritance_one']->getCardinality());
    $this->assertEquals(3, $fields['inheritance_two']->getCardinality());
  }

  /**
   * Tests the 'prepend' inheritance type.
   */
  public function testPrependInheritanceType(): void {
    $this->inheritanceOne->setType('prepend')->save();
    $this->inheritanceTwo->setType('prepend')->save();
    drupal_flush_all_caches();

    $parent = EntityTest::load($this->parent->id());
    $child = EntityTest::load($this->child->id());

    $this->assertEquals($child->inheritance_one->getValue(), array_merge($child->field_one->getValue(), $parent->field_one->getValue()));
    $this->assertEquals($child->inheritance_two->getValue(), array_merge($child->field_two->getValue(), $parent->field_two->getValue()));

    $fields = \Drupal::service('entity_field.manager')->getFieldDefinitions('entity_test', 'entity_test');
    $this->assertEquals(-1, $fields['inheritance_one']->getCardinality());
    $this->assertEquals(6, $fields['inheritance_two']->getCardinality());
  }

  /**
   * Tests the 'append' inheritance type.
   */
  public function testAppendInheritanceType(): void {
    $this->inheritanceOne->setType('append')->save();
    $this->inheritanceTwo->setType('append')->save();
    drupal_flush_all_caches();

    $parent = EntityTest::load($this->parent->id());
    $child = EntityTest::load($this->child->id());

    $this->assertEquals($child->inheritance_one->getValue(), array_merge($parent->field_one->getValue(), $child->field_one->getValue()));
    $this->assertEquals($child->inheritance_two->getValue(), array_merge($parent->field_two->getValue(), $child->field_two->getValue()));

    $fields = \Drupal::service('entity_field.manager')->getFieldDefinitions('entity_test', 'entity_test');
    $this->assertEquals(-1, $fields['inheritance_one']->getCardinality());
    $this->assertEquals(6, $fields['inheritance_two']->getCardinality());
  }

  /**
   * Tests the 'fallback' inheritance type.
   */
  public function testFallbackInheritanceType(): void {
    $this->inheritanceOne->setType('fallback')->save();
    $this->inheritanceTwo->setType('fallback')->save();
    drupal_flush_all_caches();

    $parent = EntityTest::load($this->parent->id());
    $child = EntityTest::load($this->child->id());

    $this->assertEquals($child->inheritance_one->getValue(), $child->field_one->getValue());
    $this->assertEquals($child->inheritance_two->getValue(), $child->field_two->getValue());

    $fields = \Drupal::service('entity_field.manager')->getFieldDefinitions('entity_test', 'entity_test');
    $this->assertEquals(-1, $fields['inheritance_one']->getCardinality());
    $this->assertEquals(3, $fields['inheritance_two']->getCardinality());

    $child->field_one = NULL;
    $child->field_two = NULL;
    $child->save();

    $child = \Drupal::entityTypeManager()->getStorage('entity_test')->loadUnchanged($this->child->id());

    $this->assertEquals($child->inheritance_one->getValue(), $parent->field_one->getValue());
    $this->assertEquals($child->inheritance_two->getValue(), $parent->field_two->getValue());
  }

}
