<?php

namespace Drupal\field_login;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Plugin\DefaultPluginManager;

/**
 * FieldLogin plugin manager.
 */
class FieldLoginPluginManager extends DefaultPluginManager {

  /**
   * Constructs FieldLoginPluginManager object.
   *
   * @param \Traversable $namespaces
   *   An object that implements \Traversable which contains the root paths
   *   keyed by the corresponding namespace to look for plugin implementations.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache_backend
   *   Cache backend instance to use.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler to invoke the alter hook with.
   */
  public function __construct(\Traversable $namespaces, CacheBackendInterface $cache_backend, ModuleHandlerInterface $module_handler) {
    parent::__construct(
      'Plugin/FieldLogin',
      $namespaces,
      $module_handler,
      'Drupal\field_login\FieldLoginInterface',
      'Drupal\field_login\Annotation\FieldLogin'
    );
    $this->alterInfo('field_login_info');
    $this->setCacheBackend($cache_backend, 'field_login_plugins');
  }

  /**
   * Determine if the plugin exists.
   *
   * @param string $field_name
   *  Implement the field name for user login.
   *
   * @return bool
   */
  public function hasFieldLoginPlugin(string $field_name): bool {
    foreach ($this->getDefinitions() as $plugin_id => $plugin_info) {
      if ($plugin_info['field_name'] == $field_name) {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * Obtain the user's UID from the plugin.
   *
   * @param string $field_name
   *   Implement the field name for user login.
   * @param string $inputValue
   *   The value passed in by the user login form.
   *
   * @return array
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  public function getFieldLoginPlugin(string $field_name, string $inputValue): array {
    foreach ($this->getDefinitions() as $plugin_id => $plugin_info) {
      if ($plugin_info['field_name'] == $field_name) {
        $instances = $this->createInstance($plugin_id);
        return $instances->getAccountUid($inputValue);
      }
    }
    return [];
  }

}
