<?php

namespace Drupal\field_login;

use Drupal;
use Drupal\user\UserAuthInterface;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Entity\EntityTypeManagerInterface;

/**
 * Validates user authentication credentials.
 */
class UserAuthDecorator implements UserAuthInterface {

  use DependencySerializationTrait;

  /**
   * The original user authentication service.
   *
   * @var \Drupal\user\UserAuthInterface
   */
  protected $userAuth;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The user storage.
   *
   * @var \Drupal\user\UserStorageInterface
   */
  protected $userStorage;

  /**
   * Constructs a UserAuth object.
   *
   * @param \Drupal\user\UserAuthInterface $user_auth
   *   The original user authentication service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   */
  public function __construct(UserAuthInterface $user_auth, EntityTypeManagerInterface $entity_type_manager) {
    $this->userAuth = $user_auth;
    $this->userStorage = $entity_type_manager->getStorage('user');
  }

  /**
   * {@inheritdoc}
   */
  public function authenticate($inputValue, $password) {
    $username = $inputValue;

    $fieldLoginPlugin = Drupal::service('plugin.manager.field_login');
    $login_field = Drupal::configFactory()
      ->get('field_login.settings')
      ->get('login_field');

    // Hook Field Login.
    $hook = 'field_login';
    foreach ($login_field as $field) {
      $loginName = $field === 'mail' ? filter_var($inputValue, FILTER_VALIDATE_EMAIL) : $inputValue;

      // Plugin Query User UID.
      if (!empty($fieldLoginPlugin->hasFieldLoginPlugin($field))
        && $plugin_uid = $fieldLoginPlugin->getFieldLoginPlugin($field, $inputValue)) {
        $uid = !empty($plugin_uid[0]) ? $plugin_uid[0] : key(array_flip($plugin_uid));
      }

      // HOOK Query User UID.
      if (Drupal::moduleHandler()
          ->hasImplementations($hook) && $hook_uid = Drupal::moduleHandler()
          ->invokeAll($hook, [$field, $inputValue])) {
        $uid = !empty(key($hook_uid)) ? key($hook_uid) : key(array_flip($hook_uid));
      }

      if ($query_uid = $this->userStorage->getQuery()
        ->accessCheck()
        ->condition($field, $loginName)
        ->execute()) {
        $uid = key($query_uid);
      }

      if (!empty($uid)) {
        $account = $this->userStorage->load($uid);
        $username = $account->getAccountName();
      }
    }

    return $this->userAuth->authenticate($username, $password);
  }

}
