<?php

namespace Drupal\field_login\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Set the list of allowed login fields.
 */
class FieldLoginSettingsForm extends ConfigFormBase {

  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'field_login.settings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'field_login_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('field_login.settings');

    // Exclude default field list.
    $exclude_field_list = [
      'uuid',
      'langcode',
      'preferred_langcode',
      'preferred_admin_langcode',
      'pass',
      'timezone',
      'status',
      'created',
      'changed',
      'access',
      'login',
      'init',
      'roles',
      'default_langcode',
    ];

    $exclude_fields = array_flip($exclude_field_list);

    $fields = \Drupal::service('entity_field.manager')
      ->getFieldDefinitions('user', 'user');

    $field_options = [];
    foreach ($fields as $field) {
      if (isset($exclude_fields[$field->getName()])) {
        continue;
      }
      $field_options[$field->getName()] = $field->getLabel();
    }

    $form['fields'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Login Configurations'),
      '#open' => TRUE,
    ];

    $form['fields']['login_field'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Login field'),
      '#default_value' => $config->get('login_field'),
      '#description' => $this->t('Select the fields that allow users to log in.'),
      '#options' => $field_options,
      '#required' => TRUE,
    ];

    $form['fields']['override_login_labels'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Override login form'),
      '#default_value' => $config->get('override_login_labels'),
      '#description' => $this->t('This option allows you to override the login form username title/description.'),
    ];

    $form['fields']['login_username_title'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Override label'),
      '#default_value' => $config->get('login_username_title'),
      '#states' => [
        'visible' => [
          ':input[name="override_login_labels"]' => ['checked' => TRUE],
        ],
      ],
      '#description' => $this->t('Override the username field title.'),
    ];

    $form['fields']['login_username_description'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Override description'),
      '#default_value' => $config->get('login_username_description'),
      '#states' => [
        'visible' => [
          ':input[name="override_login_labels"]' => ['checked' => TRUE],
        ],
      ],
      '#description' => $this->t('Override the username field description.'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    if (empty($form_state->getValue('login_field'))) {
      $form_state->setErrorByName(
        'login_field',
        $this->t("Please select at least one login field.")
      );
    }

    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    parent::submitForm($form, $form_state);
    $config = $this->config('field_login.settings');

    foreach ($form_state->getValues() as $field_name => $field_value) {
      if ($field_name === 'login_field') {
        $config->set($field_name, array_filter($field_value));
      }
      else {
        $config->set($field_name, $field_value);
      }
    }

    $config->save();
  }

}
