<?php

namespace Drupal\field_login;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\user\UserAuthInterface;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Flood\FloodInterface;

/**
 * Validates user authentication credentials.
 */
class UserAuthDecorator implements UserAuthInterface {

  use DependencySerializationTrait;
  use StringTranslationTrait;

  /**
   * The original user authentication service.
   *
   * @var \Drupal\user\UserAuthInterface
   */
  protected UserAuthInterface $userAuth;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * The messenger.
   *
   * @var \Drupal\field_login\FieldLoginPluginManagerInterface
   */
  protected FieldLoginPluginManagerInterface $fieldLoginManager;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The flood service.
   *
   * @var \Drupal\Core\Flood\FloodInterface
   */
  protected FloodInterface $flood;

  /**
   * The messenger.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected MessengerInterface $messenger;

  /**
   * Constructs a UserAuth object.
   *
   * @param \Drupal\user\UserAuthInterface $user_auth
   *   The original user authentication service.
   * @param \Drupal\Core\Flood\FloodInterface $flood
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *    The messenger.
   */
  public function __construct(
    UserAuthInterface $user_auth,
    EntityTypeManagerInterface $entity_type_manager,
    ModuleHandlerInterface $module_handler,
    FieldLoginPluginManagerInterface $field_login_plugin_manager,
    FloodInterface $flood,
    ConfigFactoryInterface $config_factory,
    MessengerInterface $messenger
  ) {
    $this->userAuth = $user_auth;
    $this->entityTypeManager = $entity_type_manager;
    $this->moduleHandler = $module_handler;
    $this->fieldLoginManager = $field_login_plugin_manager;
    $this->configFactory = $config_factory;
    $this->flood = $flood;
    $this->messenger = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public function authenticate($username, $password) {
    if (empty($username) || empty($password)) {
      return NULL;
    }
    $username = trim($username);
    $flood_config = $this->configFactory->get('user.flood');
    // Flood protection: this is very similar to the user login form code.
    // @see \Drupal\user\Form\UserLoginForm::validateAuthentication()
    // Do not allow any login from the current user's IP if the limit has been
    // reached. Default is 50 failed attempts allowed in one hour. This is
    // independent of the per-user limit to catch attempts from one IP to log
    // in to many different user accounts.  We have a reasonably high limit
    // since there may be only one apparent IP for all users at an institution.
    if ($this->flood->isAllowed('basic_auth.failed_login_ip', $flood_config->get('ip_limit'), $flood_config->get('ip_window'))) {
      /** @var \Drupal\user\UserInterface $account */
      if ($account = $this->userValidate($username)) {
        if ($flood_config->get('uid_only')) {
          // Register flood events based on the uid only, so they apply for any
          // IP address. This is the most secure option.
          $identifier = $account->id();
        }
        else {
          // The default identifier is a combination of uid and IP address. This
          // is less secure but more resistant to denial-of-service attacks that
          // could lock out all users with public user names.
          $identifier = $account->id() . '-' . \Drupal::request()->getClientIp();
        }
        // Don't allow login if the limit for this user has been reached.
        // Default is to allow 5 failed attempts every 6 hours.
        if ($this->flood->isAllowed('basic_auth.failed_login_user', $flood_config->get('user_limit'), $flood_config->get('user_window'), $identifier)) {
          if ($account->getAccountName() && !empty($account->isBlocked())) {
            $this->messenger->addError($this->t('The user has not been activated yet or is blocked.'));
            return FALSE;
          }

          $uid = $this->userAuth->authenticate($account->getAccountName(), $password);

          if ($uid) {
            $this->flood->clear('basic_auth.failed_login_user', $identifier);
          }
          else {
            // Register a per-user failed login event.
            $this->flood->register('basic_auth.failed_login_user', $flood_config->get('user_window'), $identifier);
          }
        }
      }
    }
    // Always register an IP-based failed login event.
    $this->flood->register('basic_auth.failed_login_ip', $flood_config->get('ip_window'));

    return $uid ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function userValidate(string $username) {
    $login_field = $this->configFactory->get('field_login.settings')->get('login_field');
    $user_storage = $this->entityTypeManager->getStorage('user');

    $account = FALSE;
    // Hook Field Login.
    $hook = 'field_login';
    foreach ($login_field as $field) {
      $loginName = $field === 'mail' ? filter_var($username, FILTER_VALIDATE_EMAIL) : $username;

      $uid = '';
      // Plugin Query User ID.
      if (!empty($this->fieldLoginManager->hasFieldLoginPlugin($field))) {
        $uid = $this->fieldLoginManager->getFieldLoginPlugin($field, $username);
      }

      // HOOK Query User ID.
      if ($this->moduleHandler->hasImplementations($hook) &&
        $hook_uid = $this->moduleHandler->invokeAll($hook, [
          $field,
          $username,
        ])) {
        $uid = $hook_uid;
      }

      // Query User ID.
      if ($query_uid = $user_storage->getQuery()
        ->accessCheck(FALSE)
        ->condition($field, $loginName)
        ->execute()) {
        $uid = $query_uid;
      }

      // Load User.
      if (!empty($uid) && $entities = $user_storage->loadMultiple($uid)) {
        /** @var \Drupal\user\UserInterface $account */
        $account = reset($entities);
      }
    }

    return $account;
  }

}
