<?php

namespace Drupal\field_login;

use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\user\UserAuthInterface;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Flood\FloodInterface;

/**
 * Validates user authentication credentials.
 */
class UserAuthDecorator implements UserAuthInterface {

  use DependencySerializationTrait;
  use StringTranslationTrait;

  /**
   * The original user authentication service.
   *
   * @var \Drupal\user\UserAuthInterface
   */
  protected UserAuthInterface $userAuth;

  /**
   * The user storage.
   *
   * @var \Drupal\field_login\UserVerificationInterface
   */
  protected UserVerificationInterface $userVerification;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The flood service.
   *
   * @var \Drupal\Core\Flood\FloodInterface
   */
  protected FloodInterface $flood;

  /**
   * The messenger.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected MessengerInterface $messenger;

  /**
   * Constructs a UserAuth object.
   *
   * @param \Drupal\user\UserAuthInterface $user_auth
   *   The original user authentication service.
   * @param \Drupal\field_login\UserVerificationInterface $user_verification
   *   This provide user authentication services.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *    The messenger.
   */
  public function __construct(
    UserAuthInterface $user_auth,
    UserVerificationInterface $user_verification,
    FloodInterface $flood,
    ConfigFactoryInterface $config_factory,
    MessengerInterface $messenger
  ) {
    $this->userAuth = $user_auth;
    $this->userVerification = $user_verification;
    $this->configFactory = $config_factory;
    $this->flood = $flood;
    $this->messenger = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public function authenticate($username, $password) {
    if (empty($username) || empty($password)) {
      return NULL;
    }
    $username = trim($username);
    $flood_config = $this->configFactory->get('user.flood');
    // Flood protection: this is very similar to the user login form code.
    // @see \Drupal\user\Form\UserLoginForm::validateAuthentication()
    // Do not allow any login from the current user's IP if the limit has been
    // reached. Default is 50 failed attempts allowed in one hour. This is
    // independent of the per-user limit to catch attempts from one IP to log
    // in to many different user accounts.  We have a reasonably high limit
    // since there may be only one apparent IP for all users at an institution.
    if ($this->flood->isAllowed('basic_auth.failed_login_ip', $flood_config->get('ip_limit'), $flood_config->get('ip_window'))) {
      /** @var \Drupal\user\UserInterface $account */
      if ($account = $this->userVerification->validate($username)) {
        if ($flood_config->get('uid_only')) {
          // Register flood events based on the uid only, so they apply for any
          // IP address. This is the most secure option.
          $identifier = $account->id();
        }
        else {
          // The default identifier is a combination of uid and IP address. This
          // is less secure but more resistant to denial-of-service attacks that
          // could lock out all users with public user names.
          $identifier = $account->id() . '-' . \Drupal::request()->getClientIp();
        }
        // Don't allow login if the limit for this user has been reached.
        // Default is to allow 5 failed attempts every 6 hours.
        if ($this->flood->isAllowed('basic_auth.failed_login_user', $flood_config->get('user_limit'), $flood_config->get('user_window'), $identifier)) {
          if ($account->getAccountName() && !empty($account->isBlocked())) {
            $this->messenger->addError($this->t('The user has not been activated yet or is blocked.'));
            return FALSE;
          }

          $uid = $this->userAuth->authenticate($account->getAccountName(), $password);

          if ($uid) {
            $this->flood->clear('basic_auth.failed_login_user', $identifier);
          }
          else {
            // Register a per-user failed login event.
            $this->flood->register('basic_auth.failed_login_user', $flood_config->get('user_window'), $identifier);
          }
        }
      }
    }
    // Always register an IP-based failed login event.
    $this->flood->register('basic_auth.failed_login_ip', $flood_config->get('ip_window'));

    return $uid ?? NULL;
  }

}
