# Field Position

A Drupal 10/11 module that provides a custom field type for selecting positions with an intuitive visual interface.

## Overview

The Field Position module allows content editors to select positions (such as top-left, center, bottom-right, etc.) using an interactive grid-based UI. This is particularly useful for:

- Positioning overlays, watermarks, or logos on images
- Placing UI elements in specific areas of a layout
- Defining anchor points for content placement
- Any scenario where spatial positioning needs to be stored

## Features

- **Visual Grid Widget**: Interactive 3x3 grid for intuitive position selection
- **Alternative Dropdown**: Optional select dropdown widget for simpler interfaces
- **Flexible Configuration**: Choose which positions are available per field
- **Multiple Display Formats**: Display as label, value, or visual icon
- **Keyboard Navigation**: Full keyboard support with arrow key navigation
- **Translatable Labels**: Position names participate in Drupal's translation system
- **Accessible**: Follows WCAG guidelines with proper ARIA attributes
- **Responsive**: Mobile-friendly interface

## Installation

1. Download and extract the module to your `modules/contrib` or `modules/custom` directory
2. Enable the module:
   ```bash
   ddev drush en field_position -y
   ```
3. Clear caches:
   ```bash
   ddev drush cr
   ```

## Usage

### Adding a Position Field

1. Navigate to **Structure → Content types → [Your Content Type] → Manage fields**
2. Click **Add field**
3. Select **Position** from the field type dropdown
4. Configure the field settings:
   - Select which positions should be available (all 9 positions by default)
5. Save the field configuration

### Widget Settings

The module provides two widget options:

#### Dropdown Select
- Traditional select dropdown

Configure the widget in **Manage form display**.

### Formatter Settings

The module provides three display formats:

1. **Label**: Displays human-readable text (e.g., "Top Left")
2. **Value**: Displays the machine name (e.g., "top-left")

Configure the formatter in **Manage display**.

## Available Positions

The module supports 9 standard positions:

- `top-left` - Top Left
- `top-center` - Top Center
- `top-right` - Top Right
- `center-left` - Center Left
- `center` - Center
- `center-right` - Center Right
- `bottom-left` - Bottom Left
- `bottom-center` - Bottom Center
- `bottom-right` - Bottom Right

## Code Examples

### Accessing Field Value in Templates

```twig
{# Get the position value #}
{% set position = node.field_position.value %}

{# Use it in a class #}
<div class="overlay overlay--{{ position }}">
  <!-- Your content -->
</div>
```

### Accessing in PHP

```php
// Load a node and get the position value.
$node = \Drupal\node\Entity\Node::load($nid);
$position = $node->get('field_position')->value;

// Check if it's a specific position.
if ($position === 'top-left') {
  // Do something.
}
```

### Programmatically Setting a Value

```php
$node->set('field_position', 'center');
$node->save();
```

## Architecture

The module follows Drupal 10/11 best practices:

- **Field Type Plugin**: `PositionItem` (src/Plugin/Field/FieldType/PositionItem.php)
- **Widget Plugin**: `PositionSelectorWidget` (src/Plugin/Field/FieldWidget/PositionSelectorWidget.php)
- **Formatter Plugin**: `PositionDefaultFormatter` (src/Plugin/Field/FieldFormatter/PositionDefaultFormatter.php)
- **Theme Hook**: `position_icon` for visual icon display
- **Asset Libraries**: Separate CSS/JS for widget and formatter

## Customization

### Adding Custom Positions

To add or adjust positions, implement `hook_field_position_positions_alter()` in a custom module:

```php
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Implements hook_field_position_positions_alter().
 */
function mymodule_field_position_positions_alter(array &$positions): void {
  $positions['custom-position'] = new TranslatableMarkup('Custom position');
}
```

After adding positions, update your CSS/JS as needed so the new options display correctly in the grid.

### Custom Styling

Override the CSS by adding your own library with higher specificity:

```css
.position-selector-grid label {
  /* Your custom styles */
}
```


## License

GPL-2.0-or-later

## Support

For bug reports and feature requests, please use the issue queue.
