/**
 * @file
 * JavaScript behaviors for the position selector widget.
 */

(function (Drupal) {
  'use strict';

  const GRID_ORDER = [
    'top-left', 'top-center', 'top-right',
    'center-left', 'center', 'center-right',
    'bottom-left', 'bottom-center', 'bottom-right',
  ];

  const INDEX_BY_POSITION = GRID_ORDER.reduce((mapping, position, index) => {
    mapping[position] = index;
    return mapping;
  }, {});

  const ARROW_KEYS = ['ArrowLeft', 'ArrowRight', 'ArrowUp', 'ArrowDown'];

  const hasPosition = (position) => Object.prototype.hasOwnProperty.call(INDEX_BY_POSITION, position);

  const moveIndex = (index, key) => {
    const row = Math.floor(index / 3);
    const col = index % 3;

    switch (key) {
      case 'ArrowLeft':
        return row * 3 + ((col + 2) % 3);
      case 'ArrowRight':
        return row * 3 + ((col + 1) % 3);
      case 'ArrowUp':
        return ((row + 2) % 3) * 3 + col;
      case 'ArrowDown':
        return ((row + 1) % 3) * 3 + col;
      default:
        return index;
    }
  };

  const findTargetPosition = (currentPosition, key, available) => {
    if (!hasPosition(currentPosition)) {
      return null;
    }

    const currentIndex = INDEX_BY_POSITION[currentPosition];
    let nextIndex = currentIndex;

    for (let attempt = 0; attempt < GRID_ORDER.length - 1; attempt += 1) {
      nextIndex = moveIndex(nextIndex, key);

      if (nextIndex === currentIndex) {
        break;
      }

      const candidate = GRID_ORDER[nextIndex];
      if (candidate && available.has(candidate)) {
        return candidate;
      }
    }

    return null;
  };

  /**
   * Enhances the position selector with keyboard navigation.
   *
   * @type {Drupal~behavior}
   */
  Drupal.behaviors.positionSelector = {
    attach: function (context) {
      const grids = context.querySelectorAll('.position-selector-grid');

      grids.forEach((grid) => {
        if (grid.dataset.positionSelectorProcessed) {
          return;
        }
        grid.dataset.positionSelectorProcessed = 'true';

        const radios = grid.querySelectorAll('input.form-radio');
        const availablePositions = new Map();

        radios.forEach((radio) => {
          const wrapper = radio.closest('[data-position]');
          if (!wrapper) {
            return;
          }

          const position = wrapper.dataset.position;
          if (!position || !hasPosition(position)) {
            return;
          }

          availablePositions.set(position, radio);
        });

        // Add keyboard navigation (arrow keys)
        radios.forEach((radio) => {
          radio.addEventListener('keydown', (event) => {
            if (!ARROW_KEYS.includes(event.key)) {
              return;
            }

            const wrapper = radio.closest('[data-position]');
            const currentPosition = wrapper ? wrapper.dataset.position : null;

            if (!currentPosition) {
              return;
            }

            const targetPosition = findTargetPosition(currentPosition, event.key, availablePositions);
            if (!targetPosition) {
              return;
            }

            const targetRadio = availablePositions.get(targetPosition);
            if (!targetRadio) {
              return;
            }

            event.preventDefault();
            targetRadio.focus();
            targetRadio.click();
          });
        });

        // Add visual feedback on hover
        const labels = grid.querySelectorAll('label');
        labels.forEach((label) => {
          label.addEventListener('mouseenter', function () {
            this.classList.add('is-hovered');
          });
          label.addEventListener('mouseleave', function () {
            this.classList.remove('is-hovered');
          });
        });
      });
    }
  };

})(Drupal);
