<?php

declare(strict_types=1);

namespace Drupal\field_position\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\field_position\Plugin\Field\FieldType\PositionItem;

/**
 * Defines the 'position_default' field formatter.
 *
 * @FieldFormatter(
 *   id = "position_default",
 *   label = @Translation("Default"),
 *   field_types = {
 *     "position",
 *   }
 * )
 */
final class PositionDefaultFormatter extends FormatterBase {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings(): array {
    return [
      'display_format' => 'label',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state): array {
    $element = parent::settingsForm($form, $form_state);

    $element['display_format'] = [
      '#type'          => 'select',
      '#title'         => $this->t('Display format'),
      '#description'   => $this->t('Choose how to display the position value.'),
      '#options'       => [
        'label' => $this->t('Label (e.g., "Top Left")'),
        'value' => $this->t('Value (e.g., "top-left")'),
      ],
      '#default_value' => $this->getSetting('display_format'),
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary(): array {
    $summary   = [];
    $summary[] = $this->t('Display format: @format', [
      '@format' => $this->getSetting('display_format'),
    ]);

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode): array {
    $elements = [];
    $format   = $this->getSetting('display_format');
    $positions = PositionItem::getPositions();

    foreach ($items as $delta => $item) {
      $value = $item->value;

      if ($value === NULL || $value === '') {
        continue;
      }

      $label = $positions[$value] ?? NULL;

      switch ($format) {
        case 'label':
          $elements[$delta] = [
            '#markup' => $label ?? $value,
          ];
          break;

        case 'value':
          $elements[$delta] = [
            '#markup' => $value,
          ];
          break;
      }
    }

    return $elements;
  }

}
