<?php

declare(strict_types=1);

namespace Drupal\field_position\Plugin\Field\FieldType;

use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemBase;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Component\Render\MarkupInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\TypedData\DataDefinition;

/**
 * Defines the 'position' field type.
 *
 * @FieldType(
 *   id = "position",
 *   label = @Translation("Position"),
 *   description = @Translation("Allows users to select a position on a grid (e.g., top-left, center, bottom-right)."),
 *   category = @Translation("General"),
 *   default_widget = "position_selector",
 *   default_formatter = "position_default"
 * )
 */
final class PositionItem extends FieldItemBase {

  /**
   * Available position options.
   *
   * @var array<string, string>
   */
  public const POSITIONS = [
    'top-left'      => 'Top Left',
    'top-center'    => 'Top Center',
    'top-right'     => 'Top Right',
    'center-left'   => 'Center Left',
    'center'        => 'Center',
    'center-right'  => 'Center Right',
    'bottom-left'   => 'Bottom Left',
    'bottom-center' => 'Bottom Center',
    'bottom-right'  => 'Bottom Right',
  ];

  /**
   * Returns the translated list of available positions.
   *
   * @return array<string, \Drupal\Core\StringTranslation\TranslatableMarkup|\Drupal\Component\Render\MarkupInterface>
   *   The translated position labels, keyed by machine name.
   */
  public static function getPositions(): array {
    $positions = [];
    foreach (self::POSITIONS as $key => $label) {
      $positions[$key] = new TranslatableMarkup($label);
    }

    // Allow other modules to alter the available positions.
    \Drupal::moduleHandler()->alter('field_position_positions', $positions);

    foreach ($positions as $key => $label) {
      if ($label instanceof TranslatableMarkup || $label instanceof MarkupInterface) {
        continue;
      }

      if (is_scalar($label) || (is_object($label) && method_exists($label, '__toString'))) {
        $positions[$key] = new TranslatableMarkup((string) $label);
      }
      else {
        unset($positions[$key]);
      }
    }

    if ($positions === []) {
      // Ensure we never return an empty list.
      foreach (self::POSITIONS as $key => $label) {
        $positions[$key] = new TranslatableMarkup($label);
      }
    }

    return $positions;
  }

  /**
   * {@inheritdoc}
   */
  public static function propertyDefinitions(FieldStorageDefinitionInterface $field_definition): array {
    $properties['value'] = DataDefinition::create('string')
      ->setLabel(t('Position value'))
      ->setRequired(TRUE);

    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public static function schema(FieldStorageDefinitionInterface $field_definition): array {
    return [
      'columns' => [
        'value' => [
          'type'     => 'varchar',
          'length'   => 32,
          'not null' => FALSE,
        ],
      ],
      'indexes' => [
        'value' => ['value'],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultFieldSettings(): array {
    return [
      'allowed_positions' => array_keys(self::POSITIONS),
    ] + parent::defaultFieldSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function fieldSettingsForm(array $form, FormStateInterface $form_state): array {
    $element = parent::fieldSettingsForm($form, $form_state);

    $element['allowed_positions'] = [
      '#type'          => 'checkboxes',
      '#title'         => $this->t('Allowed positions'),
      '#description'   => $this->t('Select which positions should be available for selection.'),
      '#options'       => self::getPositions(),
      '#default_value' => $this->getSetting('allowed_positions'),
      '#required'      => TRUE,
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function isEmpty(): bool {
    $value = $this->get('value')->getValue();
    return $value === NULL || $value === '';
  }

  /**
   * {@inheritdoc}
   */
  public static function generateSampleValue(FieldDefinitionInterface $field_definition): array {
    $positions = array_keys(self::getPositions());
    $allowed = $field_definition->getSetting('allowed_positions');

    if (is_array($allowed) && $allowed !== []) {
      $positions = array_values(array_intersect($positions, $allowed));
    }

    if ($positions === []) {
      $positions = array_keys(self::POSITIONS);
    }

    $random = array_rand($positions);
    return ['value' => $positions[$random]];
  }

  /**
   * Gets the allowed positions for this field.
   *
   * @return array<string, \Drupal\Core\StringTranslation\TranslatableMarkup|\Drupal\Component\Render\MarkupInterface>
   *   The translated allowed position options.
   */
  public function getAllowedPositions(): array {
    $positions = self::getPositions();
    $allowed = $this->getSetting('allowed_positions');

    if (!is_array($allowed) || $allowed === []) {
      return $positions;
    }

    $allowed_positions = array_intersect_key($positions, array_flip($allowed));

    return $allowed_positions ?: $positions;
  }

}
