<?php

declare(strict_types=1);

namespace Drupal\field_position\Plugin\Field\FieldWidget;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\field_position\Plugin\Field\FieldType\PositionItem;

/**
 * Defines the 'position_selector' field widget.
 *
 * @FieldWidget(
 *   id = "position_selector",
 *   label = @Translation("Position selector"),
 *   field_types = {
 *     "position",
 *   }
 * )
 */
final class PositionSelectorWidget extends WidgetBase {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings(): array {
    return [
      'display_style' => 'grid',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state): array {
    $element = parent::settingsForm($form, $form_state);

    $element['display_style'] = [
      '#type'          => 'select',
      '#title'         => $this->t('Display style'),
      '#description'   => $this->t('Choose how the position selector should be displayed.'),
      '#options'       => [
        'grid'   => $this->t('Visual grid'),
        'select' => $this->t('Dropdown select'),
      ],
      '#default_value' => $this->getSetting('display_style'),
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary(): array {
    $summary   = [];
    $summary[] = $this->t('Display style: @style', [
      '@style' => $this->getSetting('display_style'),
    ]);

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state): array {
    $item = $items[$delta];
    $field_settings = $this->getFieldSettings();
    $positions = PositionItem::getPositions();
    $allowed_positions = $field_settings['allowed_positions'] ?? NULL;

    if (is_array($allowed_positions) && $allowed_positions !== []) {
      $options = array_intersect_key($positions, array_flip($allowed_positions));
      if ($options === []) {
        $options = $positions;
      }
    }
    else {
      $options = $positions;
    }

    // Attach library if using grid display.
    if ($this->getSetting('display_style') === 'grid') {
      $element['#attached']['library'][] = 'field_position/position_selector';
    }

    $element['value'] = [
      '#type'          => $this->getSetting('display_style') === 'grid' ? 'radios' : 'select',
      '#options'       => $options,
      '#default_value' => $item->value ?? NULL,
      '#empty_value'   => '',
      '#required'      => $element['#required'] ?? FALSE,
    ];

    if (isset($element['#title'])) {
      $element['value']['#title'] = $element['#title'];
      unset($element['#title']);
    }

    if (!empty($element['#description'])) {
      $element['value']['#description'] = $element['#description'];
      unset($element['#description']);
    }

    // Add visual grid styling for grid display style.
    if ($this->getSetting('display_style') === 'grid') {
      // Add wrapper attributes to apply the grid class via theme.
      $element['value']['#wrapper_attributes']['class'][] = 'position-selector-grid';

      // Add data-position attribute to each radio wrapper.
      foreach ($options as $key => $label) {
        $element['value'][$key]['#wrapper_attributes']['data-position'] = $key;
      }
    }

    return $element;
  }

}
