<?php

declare(strict_types=1);

namespace Drupal\field_redirect\Form;

use Drupal\Core\Config\Config;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\field_redirect\Service\Utility;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Config form class.
 */
final class ConfigForm extends ConfigFormBase {

  /**
   * The module settings.
   */
  protected Config $settings;

  /**
   * {@inheritdoc}
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    TypedConfigManagerInterface $typed_config_manager,
    protected Utility $utility,
  ) {
    parent::__construct($config_factory, $typed_config_manager);
    $this->settings = $this->config('field_redirect.settings');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new self(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('field_redirect.utility'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'field_redirect_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    // Form constructor.
    $form = parent::buildForm($form, $form_state);
    $mappings = $this->settings->get('mappings');

    // @todo This will prevent errors from occurring even with old format configuration values. This implementation will be removed in the near future.
    if (\is_array($old_mappings = $this->settings->get('field_redirect.mappings'))
      && !empty($old_mappings)
    ) {
      $mappings = $this->utility->toString($old_mappings);
    }

    // Source text field.
    $form['mappings'] = [
      '#type' => 'textarea',
      '#rows' => 10,
      '#title' => $this->t('The redirect mapping settings of entity (type and bundle) and fields:'),
      '#default_value' => $mappings,
      '#placeholder' => <<<'PLACEHOLDER'
        e.g.)
        node:article|field_source_url#301,#403
        node:page|#404
        taxonomy_term:*|field_overview_url,field_wiki_url
        user|field_sns_url
        PLACEHOLDER,
      '#description' => $this->t('* Please enter one value per line in the format "entity_type_id:entity_bundle|field_machine_name#status_code".<br>* Please separate them with comma, describe them in order priority, if specifying multiple pairs of "field_machine_name" and "status_code". (*#301, #302, #303, or #307 can be specified "status_code", but if omitted then #307.)<br>* You can also specify at line end only one 4xx status. (*#403 or #404 can be specified.)'),
    ];

    // Flush all caches when saving. (* For local environment.)
    $form['drupal_flush_all_caches_when_saving'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Flush all caches when saving.'),
      '#default_value' => $this->settings->get('drupal_flush_all_caches_when_saving'),
      '#description' => $this->t("Please use this function in a local environment.<br>We do not recommend changing the configuration of this module in a production environment.<br>If you have no choice but to change it immediately, please save it after taking a backup in advance and clear the cache with the Drush command. ( <strong>drush php:eval 'drupal_flush_all_caches();'</strong> )"),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $element = $form['mappings'] ?? [];
    $mappings_string = (string) $this->utility->trim($form_state->getValue('mappings') ?: '');
    if ($mappings_string === '') {
      $form_state->setValueForElement($element, NULL);
      return;
    }
    if (!$this->utility->isValidMappings($form_state, $element, $mappings_string)) {
      $form['mappings']['#value'] = $mappings_string;
      return;
    }
    $form_state->setValueForElement($element, $mappings_string);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->settings
      ->set('mappings', $form_state->getValue('mappings'))
      ->save();
    if ($form_state->getValue('drupal_flush_all_caches_when_saving')) {
      \drupal_flush_all_caches();
    }
    parent::submitForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['field_redirect.settings'];
  }

}
