<?php

declare(strict_types=1);

namespace Drupal\field_redirect\Hook;

use Drupal\Core\Entity\ContentEntityFormInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\field_redirect\Service\Utility;

/**
 * Form hooks class.
 */
final class FormHooks {
  use StringTranslationTrait;

  public function __construct(private Utility $utility) {}

  /**
   * Implements hook_form_BASE_FORM_ID_alter().
   */
  #[Hook('form_node_form_alter')]
  #[Hook('form_taxonomy_term_form_alter')]
  #[Hook('form_user_form_alter')]
  public function baseFormIdAlter(array &$form, FormStateInterface $form_state, string $form_id): void {
    if (($form_object = $form_state->getFormObject()) instanceof ContentEntityFormInterface
      && ($entity = $form_object->getEntity()) instanceof ContentEntityInterface
      && ($redirect_fields = $this->utility->getRedirectFields($entity->getEntityTypeId(), $entity->bundle()))
    ) {
      foreach ($redirect_fields as $field_machine_name__status) {
        [$field_machine_name, $status] = $field_machine_name__status;
        if ($widget = &$form[$field_machine_name]['widget'][0] ?? FALSE) {
          $message = $this->t('<p><strong>Field Redirect Module: If this field value is valid, redirect to the field value URL with redirect status "@status".</strong></p>', [
            '@status' => $this->utility->getHttpStatusMessage($status),
          ]);
          if (\in_array(($widget['uri']['#type'] ?? ''), ['url', 'entity_autocomplete'], TRUE)) {
            $description = $widget['uri']['#description'] ?? '';
            $widget['uri']['#description'] = $message . $description;
          }
          elseif (($widget['#type'] ?? '') === 'managed_file') {
            $description = $widget['#description'] ?? '';
            $widget['#description'] = $message . $description;
          }
        }
      }
      $form['actions']['submit']['#submit'][] = [$this, 'doRedirectWhenSubmitForm'];
    }
  }

  /**
   * Do redirect when submitted form.
   */
  public function doRedirectWhenSubmitForm(array $form, FormStateInterface $form_state): void {
    if (($form_object = $form_state->getFormObject()) instanceof ContentEntityFormInterface
      && ($entity = $form_object->getEntity()) instanceof ContentEntityInterface
    ) {
      $this->utility->doRedirect('after_save', $entity);
    }
  }

}
