<?php

declare(strict_types=1);

namespace Drupal\field_redirect\Service;

use Drupal\Component\Render\MarkupInterface;
use Drupal\Core\Config\Config;
use Drupal\Core\Entity\EditorialContentEntityBase;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Messenger\MessengerTrait;
use Drupal\Core\Render\Markup;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Drupal\field\FieldStorageConfigInterface;
use Drupal\file\FileInterface;
use Drupal\user\UserInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Exception\HttpException;

/**
 * Utility service class.
 */
final class Utility {
  use MessengerTrait;
  use StringTranslationTrait;

  private const DEFAULT_REDIRECT_STATUS = '307';
  private const ENTITY_PAIR_SEPARATOR = ':';
  private const ENTITY_BUNDLE_WILDCARD = '*';
  private const PAIR_SEPARATOR = '|';
  private const FIELDS_SEPARATOR = ',';
  private const STATUS_SEPARATOR = '#';
  private const CR = "\r";
  private const LF = "\n";
  private const CRLF = self::CR . self::LF;
  private const SUPPORTED_FIELD_TYPES = [
    'link',
    'file',
    'image',
  ];

  /**
   * The error message assoc.
   */
  private array $errors = [];

  /**
   * The line number.
   */
  private int $lineNumber = 0;

  /**
   * Service constructor.
   */
  public function __construct(
    private Config $settings,
    private Request $request,
    private EntityTypeManagerInterface $entityTypeManager,
    private EntityFieldManagerInterface $entityFieldManager,
    private FileUrlGeneratorInterface $fileUrlGenerator,
  ) {}

  /**
   * Convert the mappings array to a string.
   */
  public function toString(array $mappings_array = []): string {
    $mappings_string = '';
    if (empty($mappings_array)) {
      return $mappings_string;
    }
    foreach ($mappings_array as $entity_type_id__bundle => $fields) {
      if (\str_starts_with($entity_type_id__bundle, 'user' . self::ENTITY_PAIR_SEPARATOR)) {
        $entity_type_id__bundle = 'user';
      }
      $mappings_string .= $this->implodeEntityAndFields([
        $entity_type_id__bundle,
        $this->implodeFields($fields),
      ]) . self::LF;
    }
    return $this->trim($mappings_string);
  }

  /**
   * Convert a mappings string to an array.
   */
  public function toArray(string $mappings_string = ''): array {
    $mappings_array = [];
    $mappings_string = $this->trim($mappings_string);
    if ($mappings_string === '') {
      return $mappings_array;
    }
    if ($lines = $this->trimRecursive(\explode(self::LF, $mappings_string))) {
      foreach ($lines as $entity_type_id__bundle__fields) {
        [$entity_type_id__bundle, $fields] = $this->explodeEntityAndFields($entity_type_id__bundle__fields);
        $mappings_array[$entity_type_id__bundle] = \array_map(
          fn ($field_machine_name__status) => $this->explodeFieldMachineNameAndStatus($field_machine_name__status),
          $this->explodeFields($fields),
        );
      }
    }
    return $mappings_array;
  }

  /**
   * Smart trimmer.
   */
  public function trim(string $string): string {
    $string = \str_replace(self::CRLF, self::LF, $string);
    $string = \str_replace(self::CR, self::LF, $string);
    $string = \preg_replace('/\n+/', self::LF, $string);
    if (\is_string($string)
      && $string !== ''
      && ($trimmed_string = \mb_ereg_replace('^[[:space:]]*([\s\S]*?)[[:space:]]*$', '\1', $string))
    ) {
      return $trimmed_string;
    }
    return '';
  }

  /**
   * Smart recursive trimmer.
   */
  public function trimRecursive(array $array): array {
    foreach ($array as $k => $v) {
      $array[$k] = \is_array($v) ? $this->trimRecursive($v) : $this->trim($v);
    }
    return $array;
  }

  /**
   * Validate that the mappings is correct.
   */
  public function isValidMappings(FormStateInterface $form_state, array $element, string $mappings_string = ''): bool {
    $mappings_string = $this->trim($mappings_string);
    if ($mappings_string === '') {
      return TRUE;
    }
    if ($lines = $this->trimRecursive(\explode(self::LF, $mappings_string))) {
      $this->errors = [];
      $this->lineNumber = 0;
      $duplicates_entity = [];
      foreach ($lines as $entity_type_id__bundle__fields) {
        if ($entity_type_id__bundle__fields === '') {
          continue;
        }
        ++$this->lineNumber;
        [$entity_type_id__bundle, $fields] = $this->explodeEntityAndFields($entity_type_id__bundle__fields);
        [$entity_type_id, $bundle] = $this->explodeTypeAndBundle($entity_type_id__bundle);
        if ($entity_type_id === '' || ($entity_type_id !== 'user' && $bundle === '') || $fields === '') {
          $this->mergeErrorMessage('This mapping format is incorrect.');
          continue;
        }
        $duplicates_entity[$entity_type_id__bundle] ??= 0;
        ++$duplicates_entity[$entity_type_id__bundle];
        $field_storage_definitions = $this->entityFieldManager->getFieldStorageDefinitions($entity_type_id);
        if (empty($field_storage_definitions)) {
          $this->mergeErrorMessage('The bundle name "@bundle" does not exist for entity type "@entity_type_id".', [
            '@bundle' => $bundle,
            '@entity_type_id' => $entity_type_id,
          ]);
        }
        $field_machine_names = $this->explodeFields($fields);
        $duplicates_field = [];
        $max_i = \count($field_machine_names) - 1;
        foreach ($field_machine_names as $current_i => $field_machine_name__status) {
          $duplicates_field[$field_machine_name__status] ??= 0;
          ++$duplicates_field[$field_machine_name__status];
          [$field_machine_name, $status] = $this->explodeFieldMachineNameAndStatus($field_machine_name__status);
          if (!empty($field_machine_name)) {
            $field_type = NULL;
            if ($field_storage_definition = $field_storage_definitions[$field_machine_name] ?? FALSE) {
              if ($field_storage_definition instanceof BaseFieldDefinition
                || ($field_storage_definition instanceof FieldStorageConfigInterface
                  && ($bundle === self::ENTITY_BUNDLE_WILDCARD
                    || ($field_storage_definition->getBundles()[$bundle] ?? FALSE)
                  )
                )
              ) {
                $field_type = $field_storage_definition->getType();
              }
            }
            if (!isset($field_type)) {
              if ($bundle === self::ENTITY_BUNDLE_WILDCARD) {
                $this->mergeErrorMessage('The field for "@field_machine_name" does not exist for entity type "@entity_type_id".', [
                  '@field_machine_name' => $field_machine_name,
                  '@entity_type_id' => $entity_type_id,
                ]);
              }
              else {
                $this->mergeErrorMessage('The field for "@field_machine_name" does not exist for bundle "@bundle" of entity type "@entity_type_id".', [
                  '@field_machine_name' => $field_machine_name,
                  '@bundle' => $bundle,
                  '@entity_type_id' => $entity_type_id,
                ]);
              }
            }
            elseif (!\in_array($field_type, self::SUPPORTED_FIELD_TYPES, TRUE)) {
              $this->mergeErrorMessage('The field type for "@field_machine_name" must be link, file or image.', [
                '@field_machine_name' => $field_machine_name,
              ]);
            }
          }
          if (!empty($status)) {
            if (\str_starts_with($status, '3')) {
              if ($field_machine_name === '') {
                $this->mergeErrorMessage('The field should be specified if status "#3xx" is specified.');
              }
              elseif (!\in_array($status, ['301', '302', '303', '307'], TRUE)) {
                $this->mergeErrorMessage('The specified status is invalid. Please specify "#301", "#302", "#303" or "#307".');
              }
            }
            if (\str_starts_with($status, '4')) {
              if ($field_machine_name !== '') {
                $this->mergeErrorMessage('The field cannot be specified if status "#4xx" is specified.');
              }
              elseif (!\in_array($status, ['403', '404'], TRUE)) {
                $this->mergeErrorMessage('The specified status is invalid. Please specify "#403" or "#404".');
              }
              if ($max_i !== $current_i) {
                $this->mergeErrorMessage('Status is can be specified only one at the end.');
              }
            }
          }
        }
        if (empty($duplicates_field)) {
          return FALSE;
        }
        foreach ($duplicates_field as $field => $count) {
          if ($count > 1) {
            if (\str_starts_with($field, '#4')) {
              $this->mergeErrorMessage('Multiple specification of status "#4xx" is not allowed.');
            }
            else {
              $this->mergeErrorMessage('Fields for "@field_machine_name" are specified is duplicate.', [
                '@field_machine_name' => $field,
              ]);
            }
          }
        }
      }
    }
    if (empty($duplicates_entity)) {
      return FALSE;
    }
    foreach ($duplicates_entity as $entity_type_bundle => $count) {
      if ($count > 1) {
        $this->mergeErrorMessage('Set of the entity type and bundle is duplicated. (@entity_type_bundle)', [
          '@entity_type_bundle' => $entity_type_bundle,
        ], FALSE);
      }
    }
    if (!empty($this->errors)) {
      $form_state->setError($element, Markup::create(\implode('<br>', \array_keys($this->errors))));
      return FALSE;
    }
    return TRUE;
  }

  /**
   * Redirect to the specified destination or return the status.
   */
  public function doRedirect(string $mode, EntityInterface $entity): void {
    if ($entity = self::checkEntityInstance($entity, TRUE)) {
      $target_entity_type = $entity->getEntityTypeId();
      $target_entity_bundle = $entity->bundle();
      $target_entity_id = $entity->id();

      // Check exists target field in entities bundle.
      $redirect_fields = $this->getRedirectFields($target_entity_type, $target_entity_bundle);

      $field_machine_name = '';
      $field_value = [];
      $status = '';
      foreach ($redirect_fields as $field_machine_name__status) {
        [$_field_machine_name, $status] = $field_machine_name__status;
        if ($entity->hasField($_field_machine_name) && ($_field_value = $entity->get($_field_machine_name)->getValue())) {
          $field_machine_name = $_field_machine_name;
          $field_value = $_field_value;
          break;
        }
      }
      if ($field_machine_name === '' && $status === '') {
        return;
      }

      // The "taxonomy_term" path seems to be a little different from the
      // others.
      $target_entity_type_path = ($target_entity_type === 'taxonomy_term')
        ? 'taxonomy/term'
        : $target_entity_type;

      switch ($mode) {
        case 'after_save':
          // If exist "destination" in current url parameters then this function
          // process ending here, because to give priority to "destination".
          if ($this->request->query->get('destination')) {
            return;
          }

          // Get label of entities bundle.
          $target_bundle_label = $entity->getEntityType()->getLabel();

          // 4xx.
          if (\str_starts_with($status, '4')) {
            // Display a status message descriptioning this behavior to avoid
            // editor confusion.
            $this->messenger()->addStatus($this->t('Field Redirect Module: The original behavior was that if you accessed this %target_bundle_label view directly, then return status of the @status, but because I had just edited it, was redirected to the %target_bundle_label edit form.', [
              '%target_bundle_label' => $target_bundle_label,
              '@status' => $this->getHttpStatusMessage($status),
            ]));
          }

          // Redirect.
          else {
            // Display a status message descriptioning this behavior to avoid
            // editor confusion.
            $this->messenger()->addStatus($this->t('Field Redirect Module: The original behavior was that if you accessed this %target_bundle_label view directly, would be redirect to %a_tag (@status), but since it was just edited, was redirected to the %target_bundle_label edit form.', [
              '%target_bundle_label' => $target_bundle_label,
              '%a_tag' => $this->getRedirectUrlLink($field_value),
              '@status' => $this->getHttpStatusMessage($status),
            ]));
          }

          // Return to the edit form.
          $redirect_path = '/' . $target_entity_type_path . '/' . $target_entity_id . '/edit';
          $redirect_url = Url::fromUserInput($redirect_path)->toString();
          (new RedirectResponse($redirect_url))->send();
          exit;

        case 'view':
          // 4xx.
          if (\str_starts_with($status, '4') && \ctype_digit($status)) {
            throw new HttpException((int) $status);
          }

          // Redirect.
          elseif ($redirect_url = $this->getRedirectUrl($field_value)) {
            // Avoid redirect loops.
            $site_url = $this->request->getSchemeAndHttpHost();
            $site_url__no_scheme = \preg_replace('/^[^:]+/', '', $site_url);
            $current_uri = $this->request->getRequestUri();
            $current_url__http = 'http' . $site_url__no_scheme . $current_uri;
            $current_url__https = 'https' . $site_url__no_scheme . $current_uri;
            $current_entity_uri = '/' . $target_entity_type_path . '/' . $target_entity_id;
            $current_entity_url__http = 'http' . $site_url__no_scheme . $current_entity_uri;
            $current_entity_url__https = 'https' . $site_url__no_scheme . $current_entity_uri;
            if ($redirect_url === $current_uri
              || $redirect_url === $current_url__http
              || $redirect_url === $current_url__https
              || $redirect_url === $current_entity_uri
              || $redirect_url === $current_entity_url__http
              || $redirect_url === $current_entity_url__https
            ) {
              return;
            }

            // Do redirect to $redirect_url.
            $status = $status ?: self::DEFAULT_REDIRECT_STATUS;
            (new RedirectResponse($redirect_url, (int) $status))->send();
            exit;
          }
          break;
      }
    }
  }

  /**
   * Get the HTTP status message from the HTTP status code.
   */
  public function getHttpStatusMessage(string $status_code): string {
    if ($status_code === '') {
      $status_code = self::DEFAULT_REDIRECT_STATUS;
    }
    return match ($status_code) {
      '403' => '403 Forbidden',
      '404' => '404 Not Found',
      '301' => '301 Moved Permanently',
      '302' => '302 Found',
      '303' => '303 See Other',
      '307' => '307 Temporary Redirect',
      default => '',
    };
  }

  /**
   * Get redirect fields the specified bundle.
   */
  public function getRedirectFields(string $target_entity_type, string $target_entity_bundle): array {
    $mappings = $this->toArray($this->settings->get('mappings') ?: '');

    // @todo This will prevent errors from occurring even with old format configuration values. This implementation will be removed in the near future.
    if (\is_array($old_mappings = $this->settings->get('field_redirect.mappings'))
      && !empty($old_mappings)
    ) {
      $mappings = $old_mappings;
    }

    $wildcard_entity_type = $this->implodeTypeAndBundle([
      $target_entity_type,
      self::ENTITY_BUNDLE_WILDCARD,
    ]);
    $redirect_fields = $mappings[$wildcard_entity_type] ?? [];
    $specified_entity_type = $this->implodeTypeAndBundle([
      $target_entity_type,
      $target_entity_bundle,
    ]);
    $redirect_fields += $mappings[$specified_entity_type] ?? [];
    return $redirect_fields;
  }

  /**
   * Get redirect URL.
   */
  private function getRedirectUrl(array $target_field_value): string {
    // If exist value of URL.
    if ($uri = $target_field_value[0]['uri'] ?? FALSE) {
      return Url::fromUri($uri, ['attributes' => ['target' => '_blank']])->toString();
    }

    // If exist value of target id of th file entity.
    elseif (($target_id = $target_field_value[0]['target_id'] ?? FALSE)
      && ($file = $this->entityTypeManager->getStorage('file')->load($target_id)) instanceof FileInterface
      && ($uri = $file->getFileUri())
    ) {
      return $this->fileUrlGenerator->generateAbsoluteString($uri);
    }
    return '';
  }

  /**
   * Get html "a" tag markup string.
   */
  private function getRedirectUrlLink(array $target_field_value): MarkupInterface|string {
    // If exist value of URL.
    if ($uri = $target_field_value[0]['uri'] ?? FALSE) {
      $url = Url::fromUri($uri, ['attributes' => ['target' => '_blank']]);
      return Link::fromTextAndUrl($url->toString(), $url)->toString();
    }

    // If exist value of target id of th file entity.
    elseif (($target_id = $target_field_value[0]['target_id'] ?? FALSE)
      && ($file = $this->entityTypeManager->getStorage('file')->load($target_id)) instanceof FileInterface
      && ($uri = $file->getFileUri())
    ) {
      // File URL.
      $url = $this->fileUrlGenerator->generateAbsoluteString($uri);

      // Link text.
      $text = $file->getFilename();
      if (($description = (string) ($target_field_value[0]['description'] ?? '')) !== '') {
        $text .= ' (' . $description . ')';
      }

      return Markup::create('<a href="' . $url . '" target="_blank">' . $text . '</a>');
    }
    return '';
  }

  /**
   * Add error message without duplication.
   */
  private function mergeErrorMessage(string $message, array $args = [], bool $lineNumber = TRUE): void {
    if ($lineNumber) {
      $message = ((string) $this->t('Line @lineNumber:')) . ' ' . $message;
      $args += ['@lineNumber' => $this->lineNumber];
    }
    // phpcs:ignore Drupal.Semantics.FunctionT.NotLiteralString
    $error_key = (string) (new TranslatableMarkup($message, $args));
    $this->errors[$error_key] = 1;
  }

  /**
   * Explode the "entity type + entity bundle" & "fields" pair.
   */
  private function explodeEntityAndFields(string $entity_type_id__bundle__fields): array {
    $pair = $this->trimRecursive(\explode(self::PAIR_SEPARATOR, $entity_type_id__bundle__fields, 2));
    $entity_type_id__bundle = $pair[0] ?? '';
    $fields = $pair[1] ?? '';
    return [$entity_type_id__bundle, $fields];
  }

  /**
   * Implode the "entity type + entity bundle" & "fields" pair.
   */
  private function implodeEntityAndFields(array $entity_type_id__bundle__fields): string {
    $entity_type_id__bundle__fields = $this->trimRecursive($entity_type_id__bundle__fields);
    if (empty($entity_type_id__bundle__fields)) {
      return '';
    }
    if (empty($entity_type_id__bundle__fields[1])) {
      return (string) $entity_type_id__bundle__fields[0];
    }
    return \implode(self::PAIR_SEPARATOR, $entity_type_id__bundle__fields);
  }

  /**
   * Explode the "entity type" & "entity bundle" pair.
   */
  private function explodeTypeAndBundle(string $entity_type_id__bundle): array {
    $pair = $this->trimRecursive(\explode(self::ENTITY_PAIR_SEPARATOR, $entity_type_id__bundle, 2));
    $entity_type_id = $pair[0] ?? '';
    $bundle = $pair[1] ?? '';
    return [$entity_type_id, $bundle];
  }

  /**
   * Implode the "entity type" & "entity bundle" pair.
   */
  private function implodeTypeAndBundle(array $entity_type_id__bundle): string {
    $entity_type_id__bundle = $this->trimRecursive($entity_type_id__bundle);
    if (empty($entity_type_id__bundle)) {
      return '';
    }
    if (empty($entity_type_id__bundle[1])) {
      return (string) $entity_type_id__bundle[0];
    }
    return \implode(self::ENTITY_PAIR_SEPARATOR, $entity_type_id__bundle);
  }

  /**
   * Explode the "fields".
   */
  private function explodeFields(string $fields): array {
    return $this->trimRecursive(\explode(self::FIELDS_SEPARATOR, $fields));
  }

  /**
   * Implode the "fields".
   */
  private function implodeFields(array $fields): string {
    return \implode(self::FIELDS_SEPARATOR, \array_map(
      fn ($field_machine_name__status) => $this->implodeFieldMachineNameAndStatus($field_machine_name__status),
      $fields,
    ));
  }

  /**
   * Explode the "field_machine_name" & "status" pair.
   */
  private function explodeFieldMachineNameAndStatus(string $field_machine_name__status): array {
    $pair = $this->trimRecursive(\explode(self::STATUS_SEPARATOR, $field_machine_name__status, 2));
    $field_machine_name = $pair[0] ?? '';
    $status = $pair[1] ?? '';
    return [$field_machine_name, $status];
  }

  /**
   * Implode the "field_machine_name" & "status" pair.
   */
  private function implodeFieldMachineNameAndStatus(array $field_machine_name__status): string {
    $field_machine_name__status = $this->trimRecursive($field_machine_name__status);
    if (empty($field_machine_name__status)) {
      return '';
    }
    if (empty($field_machine_name__status[1])) {
      return (string) $field_machine_name__status[0];
    }
    return \implode(self::STATUS_SEPARATOR, $field_machine_name__status);
  }

  /**
   * Validate if an entity is allowed and return that entity.
   */
  private static function checkEntityInstance(EntityInterface $entity, bool $check_status = FALSE): EditorialContentEntityBase|UserInterface|false {
    if ($entity instanceof EditorialContentEntityBase) {
      if ($check_status) {
        return $entity->isPublished() ? $entity : FALSE;
      }
      return $entity;
    }
    if ($entity instanceof UserInterface) {
      if ($check_status) {
        return $entity->isActive() ? $entity : FALSE;
      }
      return $entity;
    }
    return FALSE;
  }

}
