<?php

namespace Drupal\field_sample_value\Plugin\Field\SampleValueGenerator;

use Drupal\Core\Database\Database;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\field_sample_value\SampleValueGeneratorBase;
use Drupal\field_sample_value\SampleValueGeneratorInterface;
use Drupal\trash\TrashManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines an entity reference generator.
 *
 * @SampleValueGenerator(
 *   id = "entity_reference",
 *   label = @Translation("Entity Reference"),
 *   field_types = {
 *     "entity_reference",
 *   },
 *   weight = 0,
 * )
 */
class EntityReference extends SampleValueGeneratorBase implements SampleValueGeneratorInterface, ContainerFactoryPluginInterface {

  use StringTranslationTrait;

  public function __construct(array $configuration, $plugin_id, $plugin_definition,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected ModuleHandlerInterface $moduleHandler,
    protected TrashManager $trashManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('module_handler'),
      $container->get('trash.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function generateSampleValue(FieldItemListInterface $item_list): void {
    $settings = $item_list->getFieldDefinition()->getSettings();
    $entity_type = $settings['target_type'];
    $bundles = $settings['handler_settings']['target_bundles'] ?? NULL;

    $definition = $this->entityTypeManager->getDefinition($entity_type);

    $data_table = $definition->getDataTable() ?: $definition->getBaseTable();

    $id_key = $definition->getKey('id');
    $bundle_key = $definition->getKey('bundle');
    $published_key = $definition->getKey('published');

    $select = Database::getConnection()->select($data_table, 'e');
    $select->addField('e', $id_key);

    if ($bundle_key && $bundles) {
      $select->condition("e.$bundle_key", $bundles, 'IN');
    }

    if ($published_key) {
      $select->condition("e.$published_key", 1);
    }


    if ($this->moduleHandler->moduleExists('trash') && $this->trashManager->isEntityTypeEnabled($definition)) {
      $select->condition("e.deleted", NULL, 'IS NULL');
    }

    $select->range(0, 1);
    $select->orderRandom();

    $id = $select->execute()->fetchField();

    if (empty($id)) {
      $this->messenger()->addMessage($this->formatPlural(
        count($bundles),
        "Failed to generate @label field because there are no published @type's.",
        "Failed to generate @label field because there are no published items of type: @type's.",
        [
          '@label' => $item_list->getFieldDefinition()->getLabel(),
          '@type' => reset($bundles),
          '@types' => navigation_plus_implode($bundles),
        ]
      ));
      return;
    }

    $item_list->setValue([
      'target_id' => $id,
    ]);
  }

}
