<?php

namespace Drupal\field_sample_value\Plugin\Field\SampleValueGenerator;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\field_sample_value\SampleValueGeneratorBase;
use Drupal\field_sample_value\SampleValueGeneratorInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines an entity reference generator.
 *
 * @SampleValueGenerator(
 *   id = "entity_reference",
 *   label = @Translation("Entity Reference"),
 *   field_types = {
 *     "entity_reference",
 *   },
 *   weight = 0,
 * )
 */
class EntityReference extends SampleValueGeneratorBase implements SampleValueGeneratorInterface, ContainerFactoryPluginInterface {

  use StringTranslationTrait;

  public function __construct(array $configuration, $plugin_id, $plugin_definition,
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function generateSampleValue(FieldItemListInterface $item_list): void {
    $settings = $item_list->getFieldDefinition()->getSettings();
    $entity_type = $settings['target_type'];
    $bundles = $settings['handler_settings']['target_bundles'] ?? NULL;

    $definition = $this->entityTypeManager->getDefinition($entity_type);
    $storage = $this->entityTypeManager->getStorage($entity_type);

    // Use Entity Query for workspace-aware entity selection.
    $query = $storage->getQuery()->accessCheck(TRUE);

    if ($bundles) {
      $bundle_key = $definition->getKey('bundle');
      if ($bundle_key) {
        $query->condition($bundle_key, $bundles, 'IN');
      }
    }

    $published_key = $definition->getKey('published');
    if ($published_key) {
      $query->condition($published_key, 1);
    }

    $query->range(0, 10);

    $ids = $query->execute();

    if (empty($ids)) {
      $this->messenger()->addMessage($this->formatPlural(
        count($bundles ?? []),
        "Failed to generate @label field because there are no published @type's.",
        "Failed to generate @label field because there are no published items of type: @type's.",
        [
          '@label' => $item_list->getFieldDefinition()->getLabel(),
          '@type' => reset($bundles) ?? 'items',
        ]
      ));
      return;
    }

    $id = $ids[array_rand($ids)];

    $item_list->setValue([
      'target_id' => $id,
    ]);
  }

}
