<?php

namespace Drupal\field_style\Controller;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Logger\LoggerChannelFactory;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\file\Entity\File;
use Drupal\user\Entity\User;
use GuzzleHttp\ClientInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Exception\HttpException;

/**
 * Class UploadController.
 */
class UploadController extends ControllerBase {

  /**
   * The watchdog log.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;


  /**
   * A current user instance.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The current request.
   *
   * @var \Symfony\Component\HttpFoundation\Request
   */
  protected $currentRequest;

  /**
   * A configuration object.
   *
   * @var \Drupal\Core\Config\Config
   */
  protected $config;

  /**
   * The http client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * End-point path.
   *
   * @var string
   */
  protected $path = '/field-style/attachments';
  /**
   * The body is read from a stream.
   *
   * @var string
   */
  protected $responseBody;

  /**
   * {@inheritdoc}
   */
  public function __construct(
    LoggerChannelFactory $logger,
    ConfigFactoryInterface $config,
    AccountProxyInterface $current_user,
    Request $current_request,
    ClientInterface $http_client,
  ) {
    $this->logger = $logger;
    $this->config = $config;
    $this->currentUser = $current_user;
    $this->currentRequest = $current_request;
    $this->httpClient = $http_client;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('logger.factory'),
      $container->get('config.factory'),
      $container->get('current_user'),
      $container->get('request_stack')->getCurrentRequest(),
      $container->get('http_client'),
    );
  }

  /**
   * Handle the file upload.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   */
  public function upload() {
    try {
      $user = User::load(\Drupal::currentUser()->id());
      $file = $this->saveFile();
      $response[] = [
        'id' => (Int) $file->id(),
        'url' => $file->createFileUrl(),
      ];
    }
    catch (Exception $e) {
      $response = [
        'error' => $e->getMessage(),
      ];
      return new \Symfony\Component\HttpFoundation\JsonResponse($response, $e->getStatusCode());
    }

    $statusCode = 500;
    if (!empty($response)) {
      $statusCode = 200;
    }
    return JsonResponse::fromJsonString(json_encode($response), $statusCode);
  }

  /**
   * Handle remove file.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   */
  public function remove($fid) {
    $file = File::load($fid);
    if ($file != NULL) {
      try {
        // $file->delete();
        $response = [
          'status' => TRUE,
        ];
      }
      catch (Exception $e) {
        $response = [
          'error' => $e->getMessage(),
        ];
        return new \Symfony\Component\HttpFoundation\JsonResponse($response, $e->getStatusCode());
      }
    }

    $statusCode = 500;
    if (!empty($response)) {
      $statusCode = 200;
    }
    return JsonResponse::fromJsonString(json_encode($response), $statusCode);

  }

  /**
   * Handle list files.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   */
  public function list() {
    try {
      // $user = \Drupal\user\Entity\User::load(\Drupal::currentUser()->id());
      $response = [];
      $fids = $this->currentRequest->get('fids');

      foreach ($fids as $key => $fid) {
        $file = File::load($fid);
        if (!empty($file)) {
          $response[] = [
            'id' => $file->id(),
            'url' => $file->createFileUrl(),
          ];
        }
      }
    }
    catch (Exception $e) {
      $response = [
        'error' => $e->getMessage(),
      ];
      return new \Symfony\Component\HttpFoundation\JsonResponse($response, $e->getStatusCode());
    }

    $statusCode = 500;
    if (!empty($response)) {
      $statusCode = 200;
    }
    return JsonResponse::fromJsonString(json_encode($response), $statusCode);
  }

  /**
   * Handle save file.
   */
  protected function saveFile() {
    $files = $this->currentRequest->files;
    if ($files->count() == 0) {
      return FALSE;
    }
    $keys = $files->keys();
    $uploadedFile = $this->currentRequest->files->get($keys[0])[0];

    $fileName = $uploadedFile->getClientOriginalName();
    if (empty($fileName)) {
      throw new HttpException(400, 'File name not found');
    }

    $path = $uploadedFile->getPathName();

    $fileData = file_get_contents($path, FILE_USE_INCLUDE_PATH);
    if (FALSE === $fileData) {
      throw new HttpException(400, 'File could not be processed');
    }
    $directory = 'public://' . $this->path . '/';
    \Drupal::service('file_system')->prepareDirectory($directory, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS);
    $file = File::create([
      'uri' => $directory . $fileName,
      'status' => 1,
    ]);
    file_put_contents($file->getFileUri(), $fileData);
    $file->save();

    if (!empty($file)) {
      return $file;
    }

    return FALSE;
  }

  /**
   * Maps a file name with extension to a mimetype.
   *
   * @param string $fileName
   *   The file file name.
   *
   * @return string|null
   */
  protected function mimetypeFromFileName($fileName) {
    $splitName = explode('.', $fileName);
    if (count($splitName) < 2) {
      return NULL;
    }

    $mimetypes = [
      'jpeg' => 'image/jpeg',
      'jpg' => 'image/jpeg',
      'png' => 'image/png',
      'svg' => 'image/svg+xml',
    ];

    $extension = end($splitName);
    $extension = strtolower($extension);

    return $mimetypes[$extension] ?? NULL;
  }

}
