<?php

namespace Drupal\field_style\Form;

use Drupal\Core\Database\Connection;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class FieldStyleGoogleFontsForm.
 *
 * @package Drupal\field_style\Form
 */
class FieldStyleGoogleFontsForm extends ConfigFormBase {
  protected $database;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static($container
      ->get('database'));
  }

  /**
   * Construct a form.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   */
  public function __construct(Connection $database) {
    $this->database = $database;
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'field_style.settings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'field_style_google_fonts_form';
  }

  /**
   * {@inheritdoc}
   */

  /**
   * Form with 'add more' and 'remove' buttons.
   *
   * This example shows a button to "add more" - add another textfield, and
   * the corresponding "remove" button.
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildForm($form, $form_state);
    $config = $this->config('field_style.settings');

    $values = $config->get('settings');

    $google_font_family = $config->get('google_font_family');

    // $paragraphs = $config->get('paragraphs');
    $num_google_font_family = $form_state->get('num_google_font_family');
    if ($num_google_font_family === NULL) {
      $num_google_font_family = $google_font_family && $google_font_family ? count($google_font_family) : 1;
      $form_state->set('num_google_font_family', $num_google_font_family);
    }

    $num_google_font_family = $form_state->get('num_google_font_family');

    $form['#tree'] = TRUE;

    $form['#attached'] = [
      'library' => [
        'field_style/admin',
      ],
    ];

    $form['google_font_family'] = [
      '#type' => 'details',
      '#title' => $this->t('Google Fonts'),
      '#description' => $this->t("Enter the Google Fonts embed link as provided by [fonts.google.com]. Example: https://fonts.googleapis.com/css2?family=Poppins:wght@100;400;700&display=swap"),
      '#prefix' => '<div id="google-font-family-wrapper">',
      '#suffix' => '</div>',
      '#open' => TRUE,
    ];

    $form['google_font_family']['table'] = [
      '#type' => 'table',
      '#header' => [
        $this
          ->t('Font Family'),
        $this
          ->t('Weight'),
      ],
      '#tabledrag' => [
        [
          'action' => 'order',
          'relationship' => 'sibling',
          'group' => 'table-sort-weight',
        ],
      ],
      '#tree' => TRUE,
      '#open' => FALSE,
    ];

    for ($i = 0; $i < $num_google_font_family; $i++) {
      $form['google_font_family']['table'][$i]['#attributes']['class'][] = 'draggable';
      $form['google_font_family']['table'][$i]['#weight'] = $values['google_font_family']['table'][$i]['weight'] ?? 0;

      $form['google_font_family']['table'][$i]['name'] = [
        '#type' => 'textfield',
        '#title' => $this->t('URL Embed'),
        '#size' => 100,
        '#maxlength' => 2000,
        '#default_value' => isset($google_font_family[$i]['name']) && $google_font_family[$i]['name'] ? $google_font_family[$i]['name'] : '',
      ];
      $form['google_font_family']['table'][$i]['weight'] = [
        '#type' => 'weight',
        '#title' => $this->t('Weight for @title', ['@title' => $google_font_family[$i]['name'] ?? '']),
        '#title_display' => 'invisible',
        '#default_value' => isset($google_font_family[$i]['weight']) && $google_font_family[$i]['weight'] ?? 0,
        '#attributes' => [
          'class' => [
            'table-sort-weight',
          ],
        ],
      ];
    }

    $form['google_font_family']['actions'] = [
      '#type' => 'actions',
    ];

    $form['google_font_family']['actions']['add_name'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add Font'),
      '#submit' => ['::addOneGoogleFontFamily'],
      '#ajax' => [
        'callback' => '::addmoreGoogleFontFamilyCallback',
        'wrapper' => 'google-font-family-wrapper',
      ],
    ];

    if ($num_google_font_family > 1) {
      $form['google_font_family']['actions']['remove_name'] = [
        '#type' => 'submit',
        '#value' => $this->t('Remove Font'),
        '#submit' => ['::removeGoogleFontFamilyCallback'],
        '#ajax' => [
          'callback' => '::addmoreGoogleFontFamilyCallback',
          'wrapper' => 'google-font-family-wrapper',
        ],
      ];
    }

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save'),
    ];

    return $form;
  }

  /**
   * Submit handler for the "add-one-more" button.
   *
   * Increments the max counter and causes a rebuild.
   */
  public function addOneGoogleFontFamily(array &$form, FormStateInterface $form_state) {
    $name_field = $form_state->get('num_google_font_family');
    $add_button = $name_field + 1;
    $form_state->set('num_google_font_family', $add_button);
    // Since our buildForm() method relies on the value of 'num_google_font_family' to
    // generate 'name' form elements, we have to tell the form to rebuild. If we
    // don't do this, the form builder will not call buildForm().
    $form_state->setRebuild();
  }

  /**
   * Submit handler for the "remove one" button.
   *
   * Decrements the max counter and causes a form rebuild.
   */
  public function removeGoogleFontFamilyCallback(array &$form, FormStateInterface $form_state) {
    $name_field = $form_state->get('num_google_font_family');
    if ($name_field > 1) {
      $remove_button = $name_field - 1;
      $form_state->set('num_google_font_family', $remove_button);
    }
    // Since our buildForm() method relies on the value of 'num_google_font_family' to
    // generate 'name' form elements, we have to tell the form to rebuild. If we
    // don't do this, the form builder will not call buildForm().
    $form_state->setRebuild();
  }

  /**
   * Callback for both ajax-enabled buttons.
   *
   * Selects and returns the fieldset with the names in it.
   */
  public function addmoreGoogleFontFamilyCallback(array &$form, FormStateInterface $form_state) {
    $name_field = $form_state->get('num_google_font_family');
    $add_button = $name_field + 1;
    $form_state->set('num_google_font_family', $add_button);
    return $form['google_font_family'];
  }

  /**
   * Final submit handler.
   *
   * Reports what values were finally set.
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // $values = $form_state->getValue(['break_points', 'name']);
    $output = $this->t('Configs are been saved!');
    $this->messenger()->addMessage($output);

    $form_values = $form_state->getValues();

    if (!empty($form_values['google_font_family']['table'])) {
      usort($form_values['google_font_family']['table'], [$this, 'sortByWeight']);
    }

    $config = $this->config('field_style.settings');
    $config->set('google_font_family', $form_values['google_font_family']['table']);

    $config->save();
  }

  /**
   *
   */
  private function sortByWeight($a, $b) {
    return ($a["weight"] <= $b["weight"]) ? -1 : 1;
  }

  /**
   *
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {

  }

}
