<?php

namespace Drupal\field_style;

use Drupal\file\Entity\File;

/**
 * Field Style Service.
 */
class FieldStyle {

  /**
   * Break Points.
   */
  public static function break_points() {
    $config = \Drupal::config('field_style.settings');
    $break_ponts = $config->get('break_points');
    $break_ponts_ = [];
    if (!empty($break_ponts)) {
      foreach ($break_ponts as $key => $value) {
        if (!empty($value['icon'])) {
          $file = File::load($value['icon'][0]);
          if ($file instanceof File && file_exists($file->getFileUri())) {
            $url = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
            $break_ponts[$key]['icon'] = $url;
          }
          else {
            $break_ponts[$key]['icon'] = '';
          }
        }
        $break_ponts_[$value['name']] = $break_ponts[$key];
      }
    }

    return $break_ponts_;
  }

  /**
   * Google Fonts.
   */
  public static function google_fonts() {
    $config = \Drupal::config('field_style.settings');
    $fonts = $config->get('google_font_family');
    $fonts_use = [];
    $data = [];
    if (!empty($fonts)) {
      foreach ($fonts as $key => $f) {
        if ($f['name'] == '') {
          continue;
        }
        // Parse the URL.
        $parts = parse_url($f['name']);
        if (!isset($parts['query'])) {
          return [
            'error' => 'Invalid URL or missing query parameters.',
          ];
        }

        // Parse query parameters.
        parse_str($parts['query'], $queryParams);

        // Extract font family.
        if (!isset($queryParams['family'])) {
          return [
            'error' => 'Font family not found in the URL.',
          ];
        }
        $familyParam = $queryParams['family'];
        preg_match('/^([^:]+)/', $familyParam, $familyMatch);
        $fontFamily = $familyMatch[1] ?? NULL;

        // Extract font styles.
        $fontStyles = [];
        if (preg_match_all('/ital,([^@]+)/', $familyParam, $stylesMatch)) {
          foreach ($stylesMatch as $s) {
            if ($s[0] == 'wght') {
              $fontStyles[] = 'normal';
            }
            if ($s[0] == 'ital,wght') {
              $fontStyles[] = 'italic';
            }
          }
        }

        if (preg_match('/wght@([^&]+)/', $f['name'], $matches)) {
          $weights_section = $matches[1];

          // Split by semicolon to process each pair (e.g., 0,100 or 1,100..900)
          $pairs = explode(';', $weights_section);

          foreach ($pairs as $pair) {
            if (preg_match('/^(\d),([\d.]+(?:\.\.\d+)?)/', $pair, $match)) {
              $ital = $match[1] === '1';
              $value = $match[2];

              if (strpos($value, '..') !== FALSE) {
                // It's a range like 100..900.
                [$start, $end] = explode('..', $value);
                $start = (int) $start;
                $end = (int) $end;

                for ($i = $start; $i <= $end; $i += 100) {
                  $fontWeights[] = $ital ? $i . 'italic' : (string) $i;
                }
              }
              else {
                $fontWeights[] = $ital ? $value . 'italic' : $value;
              }
            }
          }
        }

        $data[$fontFamily] = [
          'family' => $fontFamily,
          'variants' => array_unique($fontWeights),
          'styles' => array_unique($fontStyles),
        ];
      }
    }
    return $data;
  }

  /**
   * Custom Fonts.
   */
  public static function custom_fonts() {
    $custom_fonts = [];
    $config = \Drupal::service('config.factory')->get('field_style.settings');

    $fonts = $config->get('custom_font') ?? [];
    foreach ($fonts as $key => $font) {
      $custom_fonts[$font['font_name']] = [
        'family' => $font['font_name'],
        'variants' => [],
        'files' => [],
      ];

      foreach ($font['style_weight_files']['subrows'] as $row) {
        foreach ($row['font_files'] as $key => $file) {
          $file = File::load($file);
          if ($file instanceof File) {
            $uri = $file->getFileUri();
            if (preg_match('~\.(\w+)$~', $uri, $matches)) {
              $extension = $matches[1];
              $weight_index = $row['font_weight'] . ($row['font_style'] == 'italic' ? 'italic' : '');
              if (!isset($custom_fonts[$font['font_name']]['files'][$weight_index])) {
                $custom_fonts[$font['font_name']]['files'][$weight_index] = [];
              }
              $custom_fonts[$font['font_name']]['variants'][] = $weight_index;
              $custom_fonts[$font['font_name']]['files'][$weight_index][] = \Drupal::service('file_url_generator')->generateString($uri);
            }
          }
        }
      }
    }
    return $custom_fonts;
  }

  /**
   * Global Styles.
   */
  public static function global_style() {
    $config = \Drupal::service('config.factory')->get('field_style.global_settings');
    $global_styles = $config->get('field_styles') ?? [];
    $styles = [];

    foreach ($global_styles as $key => $value) {
      $styles[$value['id']] = [
        'value' => $value['id'],
        'label' => $value['name'],
        'style' => (Array) json_decode($value['style'], TRUE),
      ];
    }

    return $styles;
  }

}
