<?php

namespace Drupal\field_style\Form;

use Drupal\Core\Database\Connection;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class FieldStyleCustomFontsForm.
 *
 * @package Drupal\field_style\Form
 */
class FieldStyleCustomFontsForm extends ConfigFormBase {
  protected $database;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static($container
      ->get('database'));
  }

  /**
   * Construct a form.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   */
  public function __construct(Connection $database) {
    $this->database = $database;
    $dir = 'public://field-style/settings';
    \Drupal::service('file_system')->prepareDirectory($dir, 1);

  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'field_style.settings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'field_style_form';
  }

  /**
   * {@inheritdoc}
   */

  /**
   * Form with 'add more' and 'remove' buttons.
   *
   * This example shows a button to "add more" - add another textfield, and
   * the corresponding "remove" button.
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildForm($form, $form_state);
    $config = $this->config('field_style.settings');

    $values = $config->get('settings');

    $form['#tree'] = TRUE;

    $form['#attached'] = [
      'library' => [
        'field_style/admin',
      ],
    ];

    $form['custom_font'] = [
      '#type' => 'details',
      '#title' => $this->t('Custom Fonts'),
      '#description' => $this->t("This form allows you to add custom fonts to the website. Users can enter the font name and upload the font files for each weight (e.g., regular, bold, italic, etc.). This enables better control over the site's visual style and supports a more personalized brand identity."),
      '#open' => TRUE,
    ];

    // Initialize the number of rows.
    $num_rows = $form_state->get('num_rows') ?? 1;

    $custom_font = $form_state->get('custom_font');
    if ($custom_font === NULL && $config->get('custom_font')) {
      $custom_font = $config->get('custom_font');
      $form_state->set('custom_font', $custom_font);
      $num_rows = count($custom_font);
      $form_state->set('num_rows', $num_rows);
    }

    $form['custom_font']['font_table'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Font Name'),
        $this->t('Styles, Weights, and Files'),
        $this->t('Actions'),
      ],
      '#prefix' => '<div id="font-table-wrapper">',
      '#suffix' => '</div>',
    ];

    for ($i = 0; $i < $num_rows; $i++) {
      $form['custom_font']['font_table'][$i]['font_name'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Font Name'),
        '#title_display' => 'invisible',
        '#default_value' => $custom_font[$i]['font_name'] ?? '',
      ];

      $num_subrows = $form_state->get("num_subrows_$i") ?? 0;

      if ($custom_font && $num_subrows == NULL) {
        $num_subrows = count($custom_font[$i]['style_weight_files']['subrows']);
        $form_state->get("num_subrows_$i", $num_subrows);
      }

      $form['custom_font']['font_table'][$i]['style_weight_files'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['style-weight-files-wrapper']],
      ];

      $form['custom_font']['font_table'][$i]['style_weight_files']['subrows'] = [
        '#type' => 'table',
        '#header' => [
          $this->t('Font Style'),
          $this->t('Font Weight'),
          $this->t('Font Files'),
        ],
      ];

      for ($j = 0; $j < $num_subrows; $j++) {
        $form['custom_font']['font_table'][$i]['style_weight_files']['subrows'][$j]['font_style'] = [
          '#type' => 'select',
          '#title' => $this->t('Font Style'),
          '#title_display' => 'invisible',
          '#default_value' => $custom_font[$i]['style_weight_files']['subrows'][$j]['font_style'],
          '#options' => [
            'normal' => $this->t('Normal'),
            'italic' => $this->t('Italic'),
            // 'oblique' => $this->t('Oblique'),
          ],
        ];

        $form['custom_font']['font_table'][$i]['style_weight_files']['subrows'][$j]['font_weight'] = [
          '#type' => 'select',
          '#title' => $this->t('Font Weight'),
          '#title_display' => 'invisible',
          '#default_value' => $custom_font[$i]['style_weight_files']['subrows'][$j]['font_weight'],
          '#options' => [
            '100' => $this->t('100 - Thin'),
            '200' => $this->t('200 - Extra Light'),
            '300' => $this->t('300 - Light'),
            '400' => $this->t('400 - Normal'),
            '500' => $this->t('500 - Medium'),
            '600' => $this->t('600 - Semi Bold'),
            '700' => $this->t('700 - Bold'),
            '800' => $this->t('800 - Extra Bold'),
            '900' => $this->t('900 - Black'),
          ],
        ];

        $form['custom_font']['font_table'][$i]['style_weight_files']['subrows'][$j]['font_files'] = [
          '#type' => 'managed_file',
          '#title' => $this->t('Font Files'),
          '#title_display' => 'invisible',
          '#upload_location' => 'public://fonts/',
          '#default_value' => $custom_font[$i]['style_weight_files']['subrows'][$j]['font_files'],
          '#upload_validators' => [
            'file_validate_extensions' => ['woff woff2 otf ttf svg'],
          ],
          '#multiple' => TRUE,
          '#required' => TRUE,
        ];

        // Add a "Remove" button for each subrow.
        $form['custom_font']['font_table'][$i]['style_weight_files']['subrows'][$j]['remove_subrow'] = [
          '#type' => 'submit',
          '#value' => $this->t('Remove'),
          '#name' => "remove_subrow_{$i}_{$j}",
          '#submit' => [[get_class($this), 'removeSubrowSubmit']],
          '#ajax' => [
            'callback' => [get_class($this), 'updateTableAjax'],
            'wrapper' => 'font-table-wrapper',
          ],
        ];
      }

      $form['custom_font']['font_table'][$i]['style_weight_files']['add_more'] = [
        '#type' => 'submit',
        '#value' => $this->t('Add More'),
        '#name' => "add_more_subrows_$i",
        '#submit' => [[get_class($this), 'addMoreSubrowsSubmit']],
        '#attributes' => ['class' => ['button--primary']],
        '#ajax' => [
          'callback' => [get_class($this), 'updateTableAjax'],
          'wrapper' => 'font-table-wrapper',
        ],
      ];

      $form['custom_font']['font_table'][$i]['actions'] = [
        '#type' => 'actions',
        'remove_row' => [
          '#type' => 'submit',
          '#value' => $this->t('Remove'),
          '#name' => "remove_row_$i",
          '#submit' => [[get_class($this), 'removeRowSubmit']],
          '#ajax' => [
            'callback' => [get_class($this), 'updateTableAjax'],
            'wrapper' => 'font-table-wrapper',
          ],
        ],
      ];
    }

    $form['custom_font']['actions'] = [
      '#type' => 'actions',
    ];

    $form['custom_font']['actions']['add_row'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add Row'),
      '#submit' => [[get_class($this), 'addRowSubmit']],
      '#attributes' => ['class' => ['button--primary']],
      '#ajax' => [
        'callback' => [get_class($this), 'updateTableAjax'],
        'wrapper' => 'font-table-wrapper',
      ],
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#attributes' => ['class' => ['button--primary']],
      '#value' => $this->t('Save'),
    ];

    return $form;
  }

  /**
   * Submit handler for adding more fonts.
   */
  public static function addMoreFontSubmit(array &$form, FormStateInterface $form_state) {
    $num_fonts = $form_state->get('num_fonts') ?? 1;
    $form_state->set('num_fonts', $num_fonts + 1);
    $form_state->set('fonts_open', TRUE);
    $form_state->setRebuild();
  }

  /**
   * AJAX callback for adding more fonts.
   */
  public static function addMoreAjax(array &$form, FormStateInterface $form_state) {
    return $form['fonts'];
  }

  /**
   *
   */
  private function _getFields($entity) {
    $fields = [];
    foreach ($entity as $key => $value) {
      if (strpos($key, 'field_') === 0) {
        $fields[$key] = $value;
      }
    }
    return $fields;
  }

  /**
   * Add a subrow to a specific row.
   */
  public static function addMoreSubrowsSubmit(array &$form, FormStateInterface $form_state) {
    $trigger = $form_state->getTriggeringElement();
    $row_index = str_replace('add_more_subrows_', '', $trigger['#name']);
    $num_subrows = $form_state->get("num_subrows_$row_index") ?? 1;
    $form_state->set("num_subrows_$row_index", $num_subrows + 1);
    $form_state->setRebuild(TRUE);
  }

  /**
   * Add a row.
   */
  public static function addRowSubmit(array &$form, FormStateInterface $form_state) {
    $num_rows = $form_state->get('num_rows') ?? 1;
    $form_state->set('num_rows', $num_rows + 1);
    $form_state->setRebuild(TRUE);
  }

  /**
   * Remove a row.
   */
  public static function removeRowSubmit(array &$form, FormStateInterface $form_state) {
    $trigger = $form_state->getTriggeringElement();
    $row_index = str_replace('remove_row_', '', $trigger['#name']);
    $num_rows = $form_state->get('num_rows') ?? 1;
    $form_state->set('num_rows', max(1, $num_rows - 1));
    $form_state->setRebuild(TRUE);
  }

  /**
   * AJAX callback to update the table.
   */
  public static function updateTableAjax(array &$form, FormStateInterface $form_state) {
    return $form['custom_font']['font_table'];
  }

  /**
   *
   */
  public static function removeSubrowSubmit(array &$form, FormStateInterface $form_state) {
    $trigger = $form_state->getTriggeringElement();
    $name_parts = explode('_', $trigger['#name']);
    $row_index = $name_parts[2];
    $subrow_index = $name_parts[3];

    // Get the current number of subrows for the row.
    $num_subrows = $form_state->get("num_subrows_$row_index") ?? 1;

    // Only reduce if there is more than one subrow.
    if ($num_subrows > 1) {
      $form_state->set("num_subrows_$row_index", $num_subrows - 1);
    }

    // Optionally, you can remove a specific index by adjusting the data structure.
    $form_state->setRebuild(TRUE);
  }

  /**
   * Final submit handler.
   *
   * Reports what values were finally set.
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // $values = $form_state->getValue(['break_points', 'name']);
    $output = $this->t('Configs are been saved!');
    $this->messenger()->addMessage($output);

    $form_values = $form_state->getValues();
    $config = $this->config('field_style.settings');

    $custom_font = $form_state->getValue('custom_font')['font_table'];
    foreach ($custom_font as $key0 => $row) {
      foreach ($row['style_weight_files']['subrows'] as $key1 => $subrow) {
        unset($custom_font[$key0]['style_weight_files']['subrows'][$key1]['remove_subrow']);
        unset($custom_font[$key0]['style_weight_files']['add_more']);
        unset($custom_font[$key0]['actions']);
        if ($subrow['font_files']) {
          foreach ($subrow['font_files'] as $file_id) {
            $file = \Drupal::entityTypeManager()->getStorage('file')->load($file_id);
            $file->setPermanent();
            $file->save();
          }
        }
      }
    }

    $config->set('custom_font', $custom_font);

    if (!empty($form_values['break_points']['table'])) {
      foreach ($form_values['break_points']['table'] as $key => $value) {
        if (!empty($value['icon'][0])) {
          if ($file_id = $value['icon'][0]) {
            $file = \Drupal::entityTypeManager()->getStorage('file')->load($file_id);
            $file->setPermanent();
            $file->save();
          }
        }
      }
    }

    $config->save();
  }

  /**
   *
   */
  private function sortByWeight($a, $b) {
    return ($a["weight"] <= $b["weight"]) ? -1 : 1;
  }

  /**
   *
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {

  }

}
