<?php

namespace Drupal\field_style\Utils;

use Drupal\field_style\FieldStyle;
use Drupal\node\NodeInterface;

/**
 * Inline Style Utility.
 */
class InlineStyle {
  public $style;
  public $class_name;
  public $raw_style;
  public $font_family;

  /**
   * Constructor.
   */
  public function __construct() {
    $this->raw_style = '';
    $this->font_family = [];
  }

  /**
   * Build inline style.
   */
  public function build(&$object_style, &$field_name, &$class_name) {
    $break_points = FieldStyle::break_points();
    $this->raw_style = '';
    $global_style = [];

    \Drupal::moduleHandler()->invokeAll('field_style_global_style', [&$global_style]);

    if (isset($object_style['globalProperty'])) {
      if (isset($object_style['globalProperty']['global_style'])) {
        $global_style = $global_style[$object_style['globalProperty']['global_style']];
        unset($object_style['globalProperty']);
        if (empty((Array) $object_style)) {
          $object_style = $global_style['style'];
        }
      }
    }

    /**
     * Go through breakpoints
     */
    foreach ($object_style as $break_point => $states) {
      $states = (Array) $states;
      if (strtolower($break_point) !== 'default' &&  isset($break_points[$break_point]['media_queries']) && $break_points[$break_point]['media_queries']) {
        $this->raw_style .= trim($break_points[$break_point]['media_queries']) . '{';
      }
      $keys_states = array_keys($states);
      if (!empty($global_style['style'][$break_point])) {
        $global_style['style'] = (Array) $global_style['style'];
        $global_style['style'][$break_point] = (Array) $global_style['style'][$break_point];
        $keys_global = array_keys($global_style['style'][$break_point]);
        $keys_states = array_merge($keys_states, $keys_global);
        $keys_states = array_unique($keys_states);
      }
      foreach ($keys_states as $index => $state) {
        $properties = [];
        if (isset($states[$state])) {
          $properties = $states[$state];
        }
        /**
         * Clear selector when state is default.
         */
        $state_ = $state == 'default' ? '' : $state;
        $root = $this->sanitizeSelector($class_name, $state_);
        $this->raw_style .= $root . " {";

        $properties = (Array) $properties;
        $properties = array_filter($properties);

        if (!empty($global_style['style'][$break_point][$state])) {
          $global_style['style'][$break_point][$state] = (Array) $global_style['style'][$break_point][$state];
          $properties = $this->mergeProperties($global_style['style'][$break_point][$state], $properties);
        }

        $font_family = NULL;
        foreach ($properties as $property => $value) {
          $v = $this->call_hook_property($property, $value, $field_name, $class_name);
          if ($property && $v) {
            $this->raw_style .= strtolower(
              preg_replace(
                ["/([A-Z]+)/", "/_([A-Z]+)([A-Z][a-z])/"],
                ["-$1", "-$1-$2"],
                lcfirst($property)
                )
                ) . ":{$v};";
            if ($property == 'fontFamily') {
              $this->font_family[$v] = [];
              $font_family = $v;
            }
            if (isset($this->font_family[$font_family]) && $property == 'fontWeight') {
              $this->font_family[$font_family]['fontWeight'] = $v;
            }
            if (isset($this->font_family[$font_family]) && $property == 'fontStyle') {
              $this->font_family[$font_family]['fontStyle'] = $v;
            }
          }
        }

        $this->raw_style .= '}';
      }

      if (strtolower($break_point) !== 'default' &&  isset($break_points[$break_point]['media_queries']) && $break_points[$break_point]['media_queries']) {
        $this->raw_style .= '}';
      }
    }

    return preg_replace('/(?:[^\r\n,{}]+)(?:,(?=[^}]*{)|\s*{[\s]*})/', '', $this->raw_style);
  }

  /**
   * Get used font families.
   */
  public function get_font_family_used() {
    return $this->font_family ?? FALSE;
  }

  /**
   * Convert property name CamelCase to Snake_case.
   */
  private function call_hook_property($property, &$value, &$field_name, &$class_name) {
    $property = strtolower(
      preg_replace(
        ["/([A-Z]+)/", "/_([A-Z]+)([A-Z][a-z])/"],
        ["_$1", "_$1_$2"],
        lcfirst($property)
        )
      );

    \Drupal::moduleHandler()->invokeAll('field_style_property', [&$property, &$value, &$field_name, &$class_name]);
    \Drupal::moduleHandler()->invokeAll('field_style_property_' . $property, [&$value, &$field_name, &$class_name]);

    return $value;
  }

  /**
   * Merge properties.
   */
  private function mergeProperties($original, $overwrite) {
    $original = array_filter($original);
    $overwrite = array_filter($overwrite);
    return array_merge($original, $overwrite);
  }

  /**
   * Sanitize selector with tokens.
   */
  private function sanitizeSelector($selector, $state) {
    $node = \Drupal::routeMatch()->getParameter('node');
    if ($node instanceof NodeInterface) {
      $token = \Drupal::token();
      $selector = $token->replace($selector, ['node' => $node]);
    }

    return str_replace('[state]', ($state == '' ? '' : ':' . $state), $selector);
  }

}
