<?php

namespace Drupal\field_style\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure global styles for the site.
 */
class FieldStyleGlobalStyleForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['field_style.global_styles'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'global_styles_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    // Load the editable configuration.
    $global_settings = \Drupal::configFactory()->getEditable('field_style.global_settings');

    $field_styles = $global_settings->get('field_styles') ?? [''];
    // dd($field_styles);
    // Use form state to manage the number of field_style items.
    if ($form_state->get('field_styles') === NULL) {
      $form_state->set('field_styles', $field_styles);
    }

    // Use form state to manage the number of field_style items.
    if ($form_state->get('num_field_styles') === NULL) {
      $form_state->set('num_field_styles', count($field_styles));
    }
    $field_styles = $form_state->get('field_styles');
    $num_field_styles = $form_state->get('num_field_styles');

    $form['field_styles'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Field Styles'),
      '#prefix' => '<div id="field-styles-wrapper">',
      '#suffix' => '</div>',
      '#tree' => TRUE,
    ];

    for ($i = 0; $i < $num_field_styles; $i++) {

      $form['field_styles'][$i]['id'] = [
        '#type' => 'hidden',
        '#default_value' => $field_styles[$i]['id'] ?? uniqid(),
      ];
      $form['field_styles'][$i]['name'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Name') . ' ',
        '#default_value' => $field_styles[$i]['name'] ?? '',
      ];
      $form['field_styles'][$i]['style'] = [
        '#type' => 'field_style',
        '#title' => $this->t('Field Style') . ' ' . ($i + 1),
        '#default_value' => $field_styles[$i]['style'] ?? '',
        '#attributes' => [],
        "#properties" => $global_settings->get('properties'),
        "#detail_states" => $global_settings->get('states'),
      ];

    }

    $form['field_styles']['actions']['add_more'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add more'),
      '#submit' => [[$this, 'addMoreSubmit']],
      '#attributes' => ['class' => ['button--primary']],
      '#ajax' => [
        'callback' => '::addMoreCallback',
        'wrapper' => 'field-styles-wrapper',
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * Ajax callback to add more fields.
   */
  public function addMoreCallback(array &$form, FormStateInterface $form_state) {
    return $form['field_styles'];
  }

  /**
   * Custom submit handler for the "Add more" button.
   */
  public function addMoreSubmit(array &$form, FormStateInterface $form_state) {
    $num_field_styles = $form_state->get('num_field_styles');
    $form_state->set('num_field_styles', $num_field_styles + 1);
    $form_state->setRebuild();
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Ensure `field_styles` is an array and not null.
    $field_styles = $form_state->getValue('field_styles');
    unset($field_styles['actions']);
    foreach ($field_styles as $key => $value) {
      if ($value['name'] == '') {
        unset($field_styles[$key]);
      }
    }

    // Re-index the array.
    $field_styles = array_values($field_styles);

    $config = \Drupal::configFactory()->getEditable('field_style.global_settings');
    $config->set('field_styles', $field_styles);
    $config->save();

    parent::submitForm($form, $form_state);
  }

}
