<?php

namespace Drupal\field_value_tracker\Drush\Commands;

use Drupal\Component\Utility\Unicode;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;
use Psr\Container\ContainerInterface;

/**
 * Drush commands for Field Value Tracker module.
 */
final class FieldValueTrackerCommands extends DrushCommands {

  /**
   * The database connection.
   */
  protected Connection $database;

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The entity field manager.
   */
  protected EntityFieldManagerInterface $entityFieldManager;

  /**
   * Constructs a FieldValueTrackerCommands object.
   */
  public function __construct(
    Connection $database,
    EntityTypeManagerInterface $entityTypeManager,
    EntityFieldManagerInterface $entityFieldManager,
  ) {
    parent::__construct();
    $this->database = $database;
    $this->entityTypeManager = $entityTypeManager;
    $this->entityFieldManager = $entityFieldManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('database'),
      $container->get('entity_type.manager'),
      $container->get('entity_field.manager'),
    );
  }

  /**
   * Update field values from production to lower environment values.
   *
   * @param string|null $target_field
   *   The target field in format entity_type.bundle.field_name (optional).
   * @param array $options
   *   The command options.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  #[CLI\Command(name: 'field-value-tracker:update', aliases: ['fvt:update'])]
  #[CLI\Argument(name: 'target_field', description: 'Target field (entity_type.bundle.field_name). If not provided, all trackers will be processed.')]
  #[CLI\Option(name: 'dry-run', description: 'Show what would be updated without making changes.')]
  #[CLI\Usage(name: 'field-value-tracker:update', description: 'Update all tracked field values')]
  #[CLI\Usage(name: 'field-value-tracker:update node.article.field_url', description: 'Update specific field values')]
  #[CLI\Usage(name: 'field-value-tracker:update --dry-run', description: 'Preview changes without updating')]
  public function updateFieldValues(?string $target_field = NULL, array $options = ['dry-run' => FALSE]): void {
    $dry_run = $options['dry-run'];

    // Safety check: Prevent execution in production environments.
    if ($this->isProductionEnvironment()) {
      $this->logger()->error('This command cannot be run in production environments. Detected production environment.');
      $this->logger()->notice('This command is intended for lower environments (dev, stage, test) only.');
      return;
    }

    // Load field value trackers.
    $query = $this->entityTypeManager->getStorage('field_value_tracker')->getQuery();
    $query->accessCheck(FALSE);

    if ($target_field) {
      $query->condition('target_field', $target_field);
    }

    $tracker_ids = $query->execute();

    if (empty($tracker_ids)) {
      if ($target_field) {
        $this->logger()->warning('No field value tracker found for: ' . $target_field);
      }
      else {
        $this->logger()->warning('No field value trackers found.');
      }
      return;
    }

    $trackers = $this->entityTypeManager->getStorage('field_value_tracker')->loadMultiple($tracker_ids);
    $total_updated = 0;

    foreach ($trackers as $tracker) {
      /** @var \Drupal\field_value_tracker\Entity\FieldValueTracker $tracker */
      $tracker_id = $tracker->id();
      $mode = $tracker->get('mode')->value;
      $field_target = $tracker->get('target_field')->target_id;
      $prod_value = $tracker->get('prod_value')->value;
      $lower_env_value = $tracker->get('lower_env_value')->value;

      $this->logger()->info("Processing tracker: {$field_target} ({$tracker_id}) - Mode: {$mode}");

      // Show which property will be updated for debugging.
      $parts = explode('.', $field_target);
      if (count($parts) === 3) {
        [$debug_entity_type, $debug_bundle, $debug_field_name] = $parts;
        $debug_field_definitions = $this->entityFieldManager->getFieldDefinitions($debug_entity_type, $debug_bundle);
        if (isset($debug_field_definitions[$debug_field_name])) {
          $debug_main_property = $debug_field_definitions[$debug_field_name]->getFieldStorageDefinition()->getMainPropertyName();
          $this->logger()->info("Field type: {$debug_field_definitions[$debug_field_name]->getType()}, Main property: {$debug_main_property}");
        }
      }

      if ($tracker->shouldReplace()) {
        $this->logger()->info("String replacement: '{$prod_value}' → '{$lower_env_value}'");
      }
      elseif ($tracker->shouldOverwrite()) {
        $this->logger()->info("Overwrite: Set all values to '{$lower_env_value}'");
      }
      else {
        throw new \LogicException('Invalid tracker mode: ' . $tracker->get('mode')->value);
      }

      // Parse the field target.
      $parts = explode('.', $field_target);
      if (count($parts) !== 3) {
        $this->logger()->error("Invalid field target format: {$field_target}. Expected: entity_type.bundle.field_name");
        continue;
      }

      [$entity_type, $bundle, $field_name] = $parts;

      try {
        if ($tracker->shouldOverwrite()) {
          $updated_count = $this->overwriteEntityFieldValues(
            $entity_type,
            $bundle,
            $field_name,
            $lower_env_value,
            $dry_run
          );
        }
        elseif ($tracker->shouldReplace()) {
          $updated_count = $this->updateEntityFieldValues(
            $entity_type,
            $bundle,
            $field_name,
            $prod_value,
            $lower_env_value,
            $dry_run
          );
        }
        else {
          throw new \LogicException('Invalid tracker mode: ' . $tracker->get('mode')->value);
        }

        $total_updated += $updated_count;

        if ($dry_run) {
          $action = $tracker->shouldOverwrite() ? 'overwrite' : "replace '{$prod_value}' with '{$lower_env_value}'";
          $this->logger()->success("Would {$action} in {$updated_count} records for {$field_target} ({$tracker_id})");
        }
        else {
          $action = $tracker->shouldOverwrite() ? 'overwrote' : "replaced '{$prod_value}' with '{$lower_env_value}'";
          $this->logger()->success("Successfully {$action} in {$updated_count} records for {$field_target} ({$tracker_id})");
        }

      }
      catch (\Exception $e) {
        $this->logger()->error("Error processing {$field_target} ({$tracker_id}): " . $e->getMessage());
      }
    }

    if ($dry_run) {
      $this->logger()->success("Dry run complete. Would perform string replacements in {$total_updated} total records.");
    }
    else {
      $this->logger()->success("String replacement complete. Updated {$total_updated} total records.");
    }
  }

  /**
   * Update field values for a specific entity type, bundle, and field.
   */
  protected function updateEntityFieldValues(
    string $entity_type,
    string $bundle,
    string $field_name,
    string $prod_value,
    string $lower_env_value,
    bool $dry_run = FALSE,
  ): int {
    // Get entity type definition.
    $entity_definition = $this->entityTypeManager->getDefinition($entity_type);
    $data_table = $entity_definition->getDataTable() ?: $entity_definition->getBaseTable();
    $bundle_key = $entity_definition->getKey('bundle');

    // Get the field definition to determine the main property.
    $field_definitions = $this->entityFieldManager->getFieldDefinitions($entity_type, $bundle);
    if (!isset($field_definitions[$field_name])) {
      throw new \Exception("Field {$field_name} does not exist for {$entity_type}.{$bundle}.");
    }

    $field_definition = $field_definitions[$field_name];
    $main_property = $field_definition->getFieldStorageDefinition()->getMainPropertyName();

    // Build the field table name and column name.
    $field_table = $entity_type . '__' . $field_name;
    $field_column = $field_name . '_' . $main_property;

    // Check if field table exists.
    if (!$this->database->schema()->tableExists($field_table)) {
      throw new \Exception("Field table {$field_table} does not exist.");
    }

    // Build query to find records containing the production value string.
    $query = $this->database->select($field_table, 'f');
    $query->addField('f', 'entity_id');
    $query->addField('f', $field_column);
    $query->condition('f.' . $field_column, '%' . $this->database->escapeLike($prod_value) . '%', 'LIKE');

    // Join with entity table to filter by bundle if needed.
    if ($bundle_key && $bundle) {
      $query->join($data_table, 'e', 'f.entity_id = e.' . $entity_definition->getKey('id'));
      $query->condition('e.' . $bundle_key, $bundle);
    }

    $results = $query->execute()->fetchAll();

    if (empty($results)) {
      return 0;
    }

    if ($dry_run) {
      return count($results);
    }

    // Use database REPLACE function to substitute the production string with
    // lower env string.
    $entity_ids = array_column($results, 'entity_id');
    $updated = $this->database->update($field_table)
      ->expression($field_column, 'REPLACE(' . $field_column . ', :prod_value, :lower_env_value)', [
        ':prod_value' => $prod_value,
        ':lower_env_value' => $lower_env_value,
      ])
      ->condition('entity_id', $entity_ids, 'IN')
      ->condition($field_column, '%' . $this->database->escapeLike($prod_value) . '%', 'LIKE')
      ->execute();

    // Clear entity storage cache for the updated entities.
    if (!empty($entity_ids)) {
      $storage = $this->entityTypeManager->getStorage($entity_type);
      $storage->resetCache($entity_ids);
    }

    return $updated;
  }

  /**
   * Overwrite entity field values with lower environment value.
   *
   * @param string $entity_type
   *   The entity type.
   * @param string $bundle
   *   The bundle.
   * @param string $field_name
   *   The field name.
   * @param string $lower_env_value
   *   The lower environment value to set.
   * @param bool $dry_run
   *   Whether to perform a dry run.
   *
   * @return int
   *   The number of entities updated.
   */
  protected function overwriteEntityFieldValues(
    string $entity_type,
    string $bundle,
    string $field_name,
    string $lower_env_value,
    bool $dry_run = FALSE,
  ): int {
    // Get entity type definition.
    $entity_definition = $this->entityTypeManager->getDefinition($entity_type);
    $data_table = $entity_definition->getDataTable() ?: $entity_definition->getBaseTable();
    $bundle_key = $entity_definition->getKey('bundle');

    // Get the field definition to determine the main property.
    $field_definitions = $this->entityFieldManager->getFieldDefinitions($entity_type, $bundle);
    if (!isset($field_definitions[$field_name])) {
      throw new \Exception("Field {$field_name} does not exist for {$entity_type}.{$bundle}.");
    }

    $field_definition = $field_definitions[$field_name];
    $main_property = $field_definition->getFieldStorageDefinition()->getMainPropertyName();

    // Build the field table name and column name.
    $field_table = $entity_type . '__' . $field_name;
    $field_column = $field_name . '_' . $main_property;

    // Check if field table exists.
    if (!$this->database->schema()->tableExists($field_table)) {
      throw new \Exception("Field table {$field_table} does not exist.");
    }

    // Build query to find all records for this entity/bundle.
    $query = $this->database->select($field_table, 'f');
    $query->addField('f', 'entity_id');

    // Join with entity table to filter by bundle if needed.
    if ($bundle_key && $bundle) {
      $query->join($data_table, 'e', 'f.entity_id = e.' . $entity_definition->getKey('id'));
      $query->condition('e.' . $bundle_key, $bundle);
    }

    $results = $query->execute()->fetchAll();

    if (empty($results)) {
      return 0;
    }

    if ($dry_run) {
      return count($results);
    }

    // Overwrite all field values for the given bundle.
    $entity_ids = array_column($results, 'entity_id');
    $updated = $this->database->update($field_table)
      ->fields([$field_column => $lower_env_value])
      ->condition('bundle', $bundle)
      ->execute();

    // Clear entity storage cache for the updated entities.
    if (!empty($entity_ids)) {
      $storage = $this->entityTypeManager->getStorage($entity_type);
      $storage->resetCache($entity_ids);
    }

    return $updated;
  }

  /**
   * List all field value trackers.
   */
  #[CLI\Command(name: 'field-value-tracker:list', aliases: ['fvt:list'])]
  #[CLI\Usage(name: 'field-value-tracker:list', description: 'List all field value trackers')]
  public function listTrackers(): void {
    $query = $this->entityTypeManager->getStorage('field_value_tracker')->getQuery();
    $query->accessCheck(FALSE);
    $tracker_ids = $query->execute();

    if (empty($tracker_ids)) {
      $this->logger()->warning('No field value trackers found.');
      return;
    }

    $trackers = $this->entityTypeManager->getStorage('field_value_tracker')->loadMultiple($tracker_ids);

    $this->logger()->notice('Field Value Trackers:');
    $this->logger()->notice('--------------------');

    foreach ($trackers as $tracker) {
      /** @var \Drupal\field_value_tracker\Entity\FieldValueTracker $tracker */
      $id = $tracker->id();
      $target_field = $tracker->get('target_field')->target_id;
      $prod_value = $tracker->get('prod_value')->value;
      $lower_env_value = $tracker->get('lower_env_value')->value;

      $this->logger()->notice("ID: {$id}");
      $this->logger()->notice("  Target Field: {$target_field}");
      $this->logger()->notice("  Production Value: " . Unicode::truncate($prod_value ?: "N/A", 50, FALSE, TRUE));
      $this->logger()->notice("  Lower Env Value: " . Unicode::truncate($lower_env_value, 50, FALSE, TRUE));
      $this->logger()->notice('');
    }
  }

  /**
   * Check if the current environment is a production environment.
   *
   * Detects production environments for both Acquia and Pantheon hosting.
   *
   * @return bool
   *   TRUE if running in production, FALSE otherwise.
   */
  protected function isProductionEnvironment(): bool {
    // Check Acquia production environment.
    // AH_SITE_ENVIRONMENT is set to 'prod' in production.
    $acquia_env = getenv('AH_SITE_ENVIRONMENT');
    if ($acquia_env === 'prod') {
      return TRUE;
    }

    // Check Pantheon production environment.
    // PANTHEON_ENVIRONMENT is set to 'live' in production.
    $pantheon_env = getenv('PANTHEON_ENVIRONMENT');
    if ($pantheon_env === 'live') {
      return TRUE;
    }

    return FALSE;
  }

}
