<?php

namespace Drupal\field_value_tracker\Plugin\EntityReferenceSelection;

use Drupal\Core\Entity\Attribute\EntityReferenceSelection;
use Drupal\Core\Entity\Plugin\EntityReferenceSelection\DefaultSelection;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Provides entity reference selection for field_config entities.
 */
#[EntityReferenceSelection(
  id: 'default:field_config_selection',
  label: new TranslatableMarkup('Field Config Selection'),
  entity_types: ['field_config'],
  group: 'default',
  weight: 5
)]
class FieldConfigSelection extends DefaultSelection {

  /**
   * {@inheritdoc}
   */
  public function getReferenceableEntities($match = NULL, $match_operator = 'CONTAINS', $limit = 0) {
    $target_type = $this->getConfiguration()['target_type'];
    $options = [];

    // Load all field configs.
    $entities = $this->entityTypeManager->getStorage($target_type)->loadMultiple();

    $count = 0;
    foreach ($entities as $entity_id => $entity) {
      $label = $entity->label();
      $id = $entity->id();

      // Search in both label and ID.
      if ($match !== NULL && $match !== '') {
        $match_lower = strtolower($match);
        if (strpos(strtolower($label), $match_lower) === FALSE &&
            strpos(strtolower($id), $match_lower) === FALSE) {
          continue;
        }
      }

      // Group by entity type for the select widget.
      $entity_type_id = $entity->getTargetEntityTypeId();
      $options[$entity_type_id][$entity_id] = $label . ' (' . $id . ')';

      $count++;
      if ($limit > 0 && $count >= $limit) {
        break;
      }
    }

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function validateReferenceableEntities(array $ids) {
    $result = [];
    if ($ids) {
      $target_type = $this->getConfiguration()['target_type'];
      $entities = $this->entityTypeManager->getStorage($target_type)->loadMultiple($ids);
      $result = array_keys($entities);
    }
    return $result;
  }

}
