<?php

declare(strict_types=1);

namespace Drupal\Tests\field_value_tracker\Kernel;

use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\field_value_tracker\Entity\FieldValueTracker;
use Drupal\KernelTests\KernelTestBase;
use Drupal\node\Entity\NodeType;

/**
 * Tests the Field Value Tracker entity CRUD operations.
 *
 * @group field_value_tracker
 */
class FieldValueTrackerTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'field',
    'field_value_tracker',
    'node',
    'options',
    'system',
    'text',
    'user',
  ];

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Test field config entity.
   *
   * @var \Drupal\field\Entity\FieldConfig
   */
  protected $fieldConfig;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('field_value_tracker');
    $this->installEntitySchema('node');
    $this->installEntitySchema('user');
    $this->installConfig(['field', 'node', 'system']);

    $this->entityTypeManager = $this->container->get('entity_type.manager');

    // Create a test content type and field.
    NodeType::create([
      'type' => 'test_content',
      'name' => 'Test Content',
    ])->save();

    FieldStorageConfig::create([
      'field_name' => 'field_test',
      'entity_type' => 'node',
      'type' => 'string_long',
    ])->save();

    $this->fieldConfig = FieldConfig::create([
      'field_name' => 'field_test',
      'entity_type' => 'node',
      'bundle' => 'test_content',
      'label' => 'Test Field',
    ]);
    $this->fieldConfig->save();
  }

  /**
   * Tests creating a field value tracker entity in replace mode.
   */
  public function testCreateFieldValueTrackerReplaceMode(): void {
    $values = [
      'mode' => FieldValueTracker::MODE_REPLACE,
      'target_field' => $this->fieldConfig->id(),
      'prod_value' => 'production.example.com',
      'lower_env_value' => 'staging.example.com',
    ];

    $tracker = FieldValueTracker::create($values);
    $this->assertInstanceOf(FieldValueTracker::class, $tracker);

    // Save the entity.
    $status = $tracker->save();
    $this->assertEquals(SAVED_NEW, $status);

    // Verify the values.
    $this->assertEquals(FieldValueTracker::MODE_REPLACE, $tracker->get('mode')->value);
    $this->assertEquals($this->fieldConfig->id(), $tracker->get('target_field')->target_id);
    $this->assertEquals('production.example.com', $tracker->get('prod_value')->value);
    $this->assertEquals('staging.example.com', $tracker->get('lower_env_value')->value);
  }

  /**
   * Tests creating a field value tracker entity in overwrite mode.
   */
  public function testCreateFieldValueTrackerOverwriteMode(): void {
    $values = [
      'mode' => FieldValueTracker::MODE_OVERWRITE,
      'target_field' => $this->fieldConfig->id(),
      'prod_value' => 'old value',
      'lower_env_value' => 'new value',
    ];

    $tracker = FieldValueTracker::create($values);
    $status = $tracker->save();
    $this->assertEquals(SAVED_NEW, $status);

    $this->assertEquals(FieldValueTracker::MODE_OVERWRITE, $tracker->get('mode')->value);
  }

  /**
   * Tests the shouldReplace() method.
   */
  public function testShouldReplace(): void {
    $tracker = FieldValueTracker::create([
      'mode' => FieldValueTracker::MODE_REPLACE,
      'target_field' => $this->fieldConfig->id(),
      'lower_env_value' => 'test',
    ]);

    $this->assertTrue($tracker->shouldReplace());
    $this->assertFalse($tracker->shouldOverwrite());
  }

  /**
   * Tests the shouldOverwrite() method.
   */
  public function testShouldOverwrite(): void {
    $tracker = FieldValueTracker::create([
      'mode' => FieldValueTracker::MODE_OVERWRITE,
      'target_field' => $this->fieldConfig->id(),
      'lower_env_value' => 'test',
    ]);

    $this->assertTrue($tracker->shouldOverwrite());
    $this->assertFalse($tracker->shouldReplace());
  }

  /**
   * Tests loading a field value tracker entity.
   */
  public function testLoadFieldValueTracker(): void {
    $tracker = FieldValueTracker::create([
      'mode' => FieldValueTracker::MODE_REPLACE,
      'target_field' => $this->fieldConfig->id(),
      'prod_value' => 'prod',
      'lower_env_value' => 'lower',
    ]);
    $tracker->save();

    $loaded = FieldValueTracker::load($tracker->id());
    $this->assertInstanceOf(FieldValueTracker::class, $loaded);
    $this->assertEquals($tracker->id(), $loaded->id());
    $this->assertEquals('prod', $loaded->get('prod_value')->value);
    $this->assertEquals('lower', $loaded->get('lower_env_value')->value);
  }

  /**
   * Tests updating a field value tracker entity.
   */
  public function testUpdateFieldValueTracker(): void {
    $tracker = FieldValueTracker::create([
      'mode' => FieldValueTracker::MODE_REPLACE,
      'target_field' => $this->fieldConfig->id(),
      'prod_value' => 'original',
      'lower_env_value' => 'original',
    ]);
    $tracker->save();

    // Update the entity.
    $tracker->set('prod_value', 'updated');
    $tracker->set('mode', FieldValueTracker::MODE_OVERWRITE);
    $status = $tracker->save();
    $this->assertEquals(SAVED_UPDATED, $status);

    // Reload and verify.
    $loaded = FieldValueTracker::load($tracker->id());
    $this->assertEquals('updated', $loaded->get('prod_value')->value);
    $this->assertEquals(FieldValueTracker::MODE_OVERWRITE, $loaded->get('mode')->value);
    $this->assertTrue($loaded->shouldOverwrite());
  }

  /**
   * Tests deleting a field value tracker entity.
   */
  public function testDeleteFieldValueTracker(): void {
    $tracker = FieldValueTracker::create([
      'mode' => FieldValueTracker::MODE_REPLACE,
      'target_field' => $this->fieldConfig->id(),
      'lower_env_value' => 'test',
    ]);
    $tracker->save();
    $id = $tracker->id();

    // Delete the entity.
    $tracker->delete();

    // Verify it's deleted.
    $loaded = FieldValueTracker::load($id);
    $this->assertNull($loaded);
  }

  /**
   * Tests the entity label.
   */
  public function testEntityLabel(): void {
    $tracker = FieldValueTracker::create([
      'mode' => FieldValueTracker::MODE_REPLACE,
      'target_field' => $this->fieldConfig->id(),
      'lower_env_value' => 'test',
    ]);
    $tracker->save();

    $expected_label = 'Field value tracker ' . $tracker->id();
    $this->assertEquals($expected_label, $tracker->label());
  }

  /**
   * Tests creating tracker without required fields.
   */
  public function testValidationRequiredFields(): void {
    // Create tracker without required lower_env_value field.
    $tracker = FieldValueTracker::create([
      'mode' => FieldValueTracker::MODE_REPLACE,
      'target_field' => $this->fieldConfig->id(),
    ]);

    $violations = $tracker->validate();
    $this->assertCount(1, $violations);
    $this->assertEquals('lower_env_value', $violations[0]->getPropertyPath());
  }

  /**
   * Tests creating tracker without required target_field.
   */
  public function testValidationRequiredTargetField(): void {
    $tracker = FieldValueTracker::create([
      'mode' => FieldValueTracker::MODE_REPLACE,
      'lower_env_value' => 'test',
    ]);

    $violations = $tracker->validate();
    $this->assertCount(1, $violations);
    $this->assertEquals('target_field', $violations[0]->getPropertyPath());
  }

  /**
   * Tests the default mode is replace.
   */
  public function testDefaultMode(): void {
    $tracker = FieldValueTracker::create([
      'target_field' => $this->fieldConfig->id(),
      'lower_env_value' => 'test',
    ]);

    // Default mode should be replace.
    $this->assertEquals(FieldValueTracker::MODE_REPLACE, $tracker->get('mode')->value);
    $this->assertTrue($tracker->shouldReplace());
  }

  /**
   * Tests entity reference to field config.
   */
  public function testTargetFieldReference(): void {
    $tracker = FieldValueTracker::create([
      'mode' => FieldValueTracker::MODE_REPLACE,
      'target_field' => $this->fieldConfig->id(),
      'lower_env_value' => 'test',
    ]);
    $tracker->save();

    // Load the referenced field config.
    $referenced_field = $tracker->get('target_field')->entity;
    $this->assertInstanceOf(FieldConfig::class, $referenced_field);
    $this->assertEquals($this->fieldConfig->id(), $referenced_field->id());
    $this->assertEquals('Test Field', $referenced_field->label());
  }

}
