<?php

namespace Drupal\field_value_tracker\Entity;

use Drupal\Core\Entity\Attribute\ContentEntityType;
use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Defines the Field Value Tracker entity.
 */
#[ContentEntityType(
  id: 'field_value_tracker',
  label: new TranslatableMarkup('Field Value Tracker'),
  base_table: 'field_value_tracker',
  entity_keys: [
    'id' => 'id',
    'uuid' => 'uuid',
  ],
  handlers: [
    'form' => [
      'default' => 'Drupal\field_value_tracker\Form\FieldValueTrackerForm',
      'add' => 'Drupal\field_value_tracker\Form\FieldValueTrackerForm',
      'edit' => 'Drupal\field_value_tracker\Form\FieldValueTrackerForm',
      'delete' => 'Drupal\Core\Entity\ContentEntityDeleteForm',
    ],
    'route_provider' => [
      'html' => 'Drupal\Core\Entity\Routing\AdminHtmlRouteProvider',
    ],
    'access' => 'Drupal\Core\Entity\EntityAccessControlHandler',
    'views_data' => 'Drupal\views\EntityViewsData',
    'list_builder' => 'Drupal\field_value_tracker\FieldValueTrackerListBuilder',
  ],
  links: [
    'canonical' => '/admin/config/development/field-value-tracker/{field_value_tracker}',
    'add-form' => '/admin/config/development/field-value-tracker/add',
    'edit-form' => '/admin/config/development/field-value-tracker/{field_value_tracker}/edit',
    'delete-form' => '/admin/config/development/field-value-tracker/{field_value_tracker}/delete',
    'collection' => '/admin/config/development/field-value-tracker',
  ],
  admin_permission: 'administer field value tracker',
)]
class FieldValueTracker extends ContentEntityBase implements FieldValueTrackerInterface {

  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  public function label(): string {
    return $this->t('Field value tracker @id', ['@id' => $this->id()]);
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['mode'] = BaseFieldDefinition::create('list_string')
      ->setLabel(t('Mode'))
      ->setDescription(t('The mode for field value updates'))
      ->setRequired(TRUE)
      ->setDefaultValue(static::MODE_REPLACE)
      ->setSetting('allowed_values', [
        static::MODE_REPLACE => 'Replace - Replace substring in field value',
        static::MODE_OVERWRITE => 'Overwrite - Completely replace field value',
      ])
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => -5,
      ])
      ->setDisplayOptions('form', [
        'type' => 'options_select',
        'weight' => -5,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['target_field'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Field Target'))
      ->setDescription(t('Reference to the field configuration (e.g., node.article.field_url)'))
      ->setRequired(TRUE)
      ->setSetting('target_type', 'field_config')
      ->setSetting('handler', 'default:field_config_selection')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'entity_reference_label',
        'weight' => -4,
      ])
      ->setDisplayOptions('form', [
        'type' => 'field_config_autocomplete',
        'weight' => -4,
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => '60',
          'placeholder' => 'Start typing field label or ID...',
        ],
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['prod_value'] = BaseFieldDefinition::create('string_long')
      ->setLabel(t('Production Value'))
      ->setDescription(t('The field value from production environment'))
      ->setRequired(FALSE)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => -3,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textarea',
        'weight' => -3,
        'settings' => [
          'rows' => 4,
        ],
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['lower_env_value'] = BaseFieldDefinition::create('string_long')
      ->setLabel(t('Lower Environment Value'))
      ->setDescription(t('The field value from lower environment (staging, development, etc.)'))
      ->setRequired(TRUE)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => -2,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textarea',
        'weight' => -2,
        'settings' => [
          'rows' => 4,
        ],
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    return $fields;
  }

  /**
   * {@inheritdoc}
   */
  public function shouldReplace(): bool {
    return $this->get('mode')->value === static::MODE_REPLACE;
  }

  /**
   * {@inheritdoc}
   */
  public function shouldOverwrite(): bool {
    return $this->get('mode')->value === static::MODE_OVERWRITE;
  }

}
