<?php

namespace Drupal\field_value_tracker\Form;

use Drupal\Core\Entity\ContentEntityForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\field_value_tracker\Entity\FieldValueTrackerInterface;

/**
 * Form controller for Field Value Tracker edit forms.
 */
class FieldValueTrackerForm extends ContentEntityForm {

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form = parent::form($form, $form_state);

    // Add form states to hide prod_value when mode is overwrite.
    if (isset($form['prod_value'])) {
      $form['prod_value']['#states'] = [
        'visible' => [
          ':input[name="mode"]' => ['value' => 'replace'],
        ],
        'required' => [
          ':input[name="mode"]' => ['value' => 'replace'],
        ],
      ];
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    // Ensure `prod_value` is not empty when mode is replace.
    if ($form_state->getValue(['mode', 0, 'value']) === FieldValueTrackerInterface::MODE_REPLACE
      && empty($form_state->getValue(['prod_value', 0, 'value']))) {
      $form_state->setErrorByName('prod_value', $this->t('The production value cannot be empty for replace mode.'));
    }
    elseif ($form_state->getValue(['mode', 0, 'value']) === FieldValueTrackerInterface::MODE_OVERWRITE
      && !empty($form_state->getValue(['prod_value', 0, 'value']))) {
      $form_state->setErrorByName('prod_value', $this->t('The production value should be empty for overwrite mode.'));
    }
    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    $entity = $this->entity;
    $status = parent::save($form, $form_state);

    $field_label = $entity->get('target_field')->target_id;

    switch ($status) {
      case SAVED_NEW:
        $edit_link = Link::fromTextAndUrl(
          $this->t('edit it'),
          $entity->toUrl('edit-form')
        )->toString();

        $this->messenger()->addStatus($this->t('Created the @label Field Value Tracker. You can @edit_link.', [
          '@label' => $field_label,
          '@edit_link' => $edit_link,
        ]));
        break;

      default:
        $this->messenger()->addStatus($this->t('Saved the @label Field Value Tracker.', [
          '@label' => $field_label,
        ]));
    }

    $form_state->setRedirect('entity.field_value_tracker.collection');
    return $status;
  }

}
