<?php

namespace Drupal\field_value_tracker\Plugin\Field\FieldWidget;

use Drupal\Core\Field\Attribute\FieldWidget;
use Drupal\Core\Field\Plugin\Field\FieldWidget\EntityReferenceAutocompleteWidget;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Plugin implementation of the 'field_config_autocomplete' widget.
 */
#[FieldWidget(
  id: 'field_config_autocomplete',
  label: new TranslatableMarkup('Field Config Autocomplete'),
  description: new TranslatableMarkup('Autocomplete widget that shows both field label and ID.'),
  field_types: [
    'entity_reference',
  ]
)]
class FieldConfigAutocompleteWidget extends EntityReferenceAutocompleteWidget {

  /**
   * {@inheritdoc}
   */
  protected function getEntityLabels(array $entities) {
    $labels = [];

    foreach ($entities as $entity_id => $entity) {
      $label = $entity->label();
      $id = $entity->id();

      // Format: "Label (entity.bundle.field_name)".
      // Core's EntityAutocomplete::extractEntityIdFromAutocompleteInput()
      // expects format: "label (id)" with a space before the parenthesis.
      $labels[$entity_id] = $label . ' (' . $id . ')';
    }

    return $labels;
  }

  /**
   * {@inheritdoc}
   */
  public function massageFormValues(array $values, array $form, FormStateInterface $form_state) {
    // Extract entity IDs from autocomplete format "Label (entity.id)".
    // This is needed because field_config IDs contain dots which may interfere
    // with core's extraction, or the custom label format causes issues.
    foreach ($values as &$value) {
      if (!empty($value['target_id']) && is_string($value['target_id'])) {
        // If the value contains parentheses, extract the ID from the last set.
        if (preg_match('/\(([^)]+)\)$/', $value['target_id'], $matches)) {
          $value['target_id'] = $matches[1];
        }
      }
    }

    return parent::massageFormValues($values, $form, $form_state);
  }

}
