<?php

declare(strict_types=1);

namespace Drupal\Tests\field_value_tracker\Kernel;

use Drush\Log\DrushLoggerManager;
use Psr\Log\AbstractLogger;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\field_value_tracker\Drush\Commands\FieldValueTrackerCommands;
use Drupal\field_value_tracker\Entity\FieldValueTracker;
use Drupal\KernelTests\KernelTestBase;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;

/**
 * Tests the Field Value Tracker Drush commands.
 *
 * @group field_value_tracker
 */
class FieldValueTrackerCommandsTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'field',
    'field_value_tracker',
    'node',
    'options',
    'system',
    'text',
    'user',
  ];

  /**
   * The Drush commands service.
   *
   * @var \Drupal\field_value_tracker\Drush\Commands\FieldValueTrackerCommands
   */
  protected $commands;

  /**
   * Test logger to capture command output.
   *
   * @var object
   */
  protected $testLogger;

  /**
   * Test field config entity.
   *
   * @var \Drupal\field\Entity\FieldConfig
   */
  protected $fieldConfig;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('field_value_tracker');
    $this->installEntitySchema('node');
    $this->installEntitySchema('user');
    $this->installSchema('node', ['node_access']);
    $this->installConfig(['field', 'node', 'system']);

    // Create a test content type and field.
    NodeType::create([
      'type' => 'test_content',
      'name' => 'Test Content',
    ])->save();

    FieldStorageConfig::create([
      'field_name' => 'field_test_url',
      'entity_type' => 'node',
      'type' => 'string_long',
    ])->save();

    $this->fieldConfig = FieldConfig::create([
      'field_name' => 'field_test_url',
      'entity_type' => 'node',
      'bundle' => 'test_content',
      'label' => 'Test URL Field',
    ]);
    $this->fieldConfig->save();

    // Initialize the Drush commands service.
    $this->commands = FieldValueTrackerCommands::create($this->container);

    // Set up a DrushLoggerManager with an anonymous test logger.
    $this->testLogger = new class() extends AbstractLogger {

      /**
       * Captured log messages.
       */
      public array $messages = [];

      /**
       * {@inheritdoc}
       */
      public function log($level, $message, array $context = []): void {
        $this->messages[] = [
          'level' => $level,
          'message' => (string) $message,
          'context' => $context,
        ];
      }

      /**
       * Check if a message was logged.
       */
      public function hasMessage(string $needle): bool {
        foreach ($this->messages as $entry) {
          if (str_contains($entry['message'], $needle)) {
            return TRUE;
          }
        }
        return FALSE;
      }

    };

    $logger_manager = new DrushLoggerManager();
    $logger_manager->add('test', $this->testLogger);
    $this->commands->setLogger($logger_manager);
  }

  /**
   * Tests the list command with no trackers.
   */
  public function testListCommandEmpty(): void {
    $this->commands->listTrackers();

    // Verify a warning was logged about no trackers found.
    $this->assertTrue(
      $this->testLogger->hasMessage('No field value trackers found'),
      'Warning message logged when no trackers exist'
    );
  }

  /**
   * Tests the list command with trackers.
   */
  public function testListCommandWithTrackers(): void {
    // Create a tracker.
    FieldValueTracker::create([
      'mode' => FieldValueTracker::MODE_REPLACE,
      'target_field' => $this->fieldConfig->id(),
      'prod_value' => 'production.example.com',
      'lower_env_value' => 'staging.example.com',
    ])->save();

    // Execute the list command.
    $this->commands->listTrackers();

    // Verify output contains expected information.
    $this->assertTrue(
      $this->testLogger->hasMessage('Field Value Trackers'),
      'Header message logged'
    );
    $this->assertTrue(
      $this->testLogger->hasMessage($this->fieldConfig->id()),
      'Target field ID logged'
    );
    $this->assertTrue(
      $this->testLogger->hasMessage('production.example.com'),
      'Production value logged'
    );
    $this->assertTrue(
      $this->testLogger->hasMessage('staging.example.com'),
      'Lower env value logged'
    );
  }

  /**
   * Tests the update command in replace mode.
   */
  public function testUpdateCommandReplaceMode(): void {
    // Create test nodes with production URLs.
    $node1 = Node::create([
      'type' => 'test_content',
      'title' => 'Test Node 1',
      'field_test_url' => 'https://production.example.com/page1',
    ]);
    $node1->save();

    $node2 = Node::create([
      'type' => 'test_content',
      'title' => 'Test Node 2',
      'field_test_url' => 'https://production.example.com/page2',
    ]);
    $node2->save();

    $node3 = Node::create([
      'type' => 'test_content',
      'title' => 'Test Node 3',
      'field_test_url' => 'https://other.example.com/page3',
    ]);
    $node3->save();

    // Create a tracker in replace mode.
    $tracker = FieldValueTracker::create([
      'mode' => FieldValueTracker::MODE_REPLACE,
      'target_field' => $this->fieldConfig->id(),
      'prod_value' => 'production.example.com',
      'lower_env_value' => 'staging.example.com',
    ]);
    $tracker->save();

    // Execute the update command.
    $this->commands->updateFieldValues(NULL, ['dry-run' => FALSE]);

    // Reload nodes and verify updates.
    $node1 = Node::load($node1->id());
    $node2 = Node::load($node2->id());
    $node3 = Node::load($node3->id());

    $this->assertEquals('https://staging.example.com/page1', $node1->get('field_test_url')->value);
    $this->assertEquals('https://staging.example.com/page2', $node2->get('field_test_url')->value);
    $this->assertEquals('https://other.example.com/page3', $node3->get('field_test_url')->value);

    // Verify success message was logged.
    $this->assertTrue(
      $this->testLogger->hasMessage('Successfully'),
      'Success message logged after update'
    );
    $this->assertTrue(
      $this->testLogger->hasMessage('2 records'),
      'Correct number of updated records logged'
    );
  }

  /**
   * Tests the update command in overwrite mode.
   */
  public function testUpdateCommandOverwriteMode(): void {
    // Create test nodes with various URLs.
    $node1 = Node::create([
      'type' => 'test_content',
      'title' => 'Test Node 1',
      'field_test_url' => 'https://production.example.com/page1',
    ]);
    $node1->save();

    $node2 = Node::create([
      'type' => 'test_content',
      'title' => 'Test Node 2',
      'field_test_url' => 'https://different.example.com/page2',
    ]);
    $node2->save();

    // Create a tracker in overwrite mode.
    $tracker = FieldValueTracker::create([
      'mode' => FieldValueTracker::MODE_OVERWRITE,
      'target_field' => $this->fieldConfig->id(),
      'lower_env_value' => 'https://staging.example.com',
    ]);
    $tracker->save();

    // Execute the update command.
    $this->commands->updateFieldValues(NULL, ['dry-run' => FALSE]);

    // Reload nodes and verify all values are overwritten.
    $node1 = Node::load($node1->id());
    $node2 = Node::load($node2->id());

    $this->assertEquals('https://staging.example.com', $node1->get('field_test_url')->value);
    $this->assertEquals('https://staging.example.com', $node2->get('field_test_url')->value);

    // Verify overwrite mode was used.
    $this->assertTrue(
      $this->testLogger->hasMessage('Overwrite'),
      'Overwrite mode message logged'
    );
    $this->assertTrue(
      $this->testLogger->hasMessage('Successfully overwrote'),
      'Success message indicates overwrite operation'
    );
  }

  /**
   * Tests the update command with dry run mode.
   */
  public function testUpdateCommandDryRun(): void {
    // Create a test node.
    $node = Node::create([
      'type' => 'test_content',
      'title' => 'Test Node',
      'field_test_url' => 'https://production.example.com/page',
    ]);
    $node->save();

    $original_value = $node->get('field_test_url')->value;

    // Create a tracker.
    $tracker = FieldValueTracker::create([
      'mode' => FieldValueTracker::MODE_REPLACE,
      'target_field' => $this->fieldConfig->id(),
      'prod_value' => 'production.example.com',
      'lower_env_value' => 'staging.example.com',
    ]);
    $tracker->save();

    // Execute the update command in dry-run mode.
    $this->commands->updateFieldValues(NULL, ['dry-run' => TRUE]);

    // Reload node and verify value is unchanged.
    $node = Node::load($node->id());
    $this->assertEquals($original_value, $node->get('field_test_url')->value);

    // Verify dry-run messages were logged.
    $this->assertTrue(
      $this->testLogger->hasMessage('Would replace'),
      'Dry-run message uses conditional language'
    );
    $this->assertTrue(
      $this->testLogger->hasMessage('Dry run complete'),
      'Dry-run completion message logged'
    );
  }

  /**
   * Tests the update command with specific target field.
   */
  public function testUpdateCommandWithSpecificTarget(): void {
    // Create another field.
    FieldStorageConfig::create([
      'field_name' => 'field_another_url',
      'entity_type' => 'node',
      'type' => 'string_long',
    ])->save();

    $another_field = FieldConfig::create([
      'field_name' => 'field_another_url',
      'entity_type' => 'node',
      'bundle' => 'test_content',
      'label' => 'Another URL Field',
    ]);
    $another_field->save();

    // Create nodes with values in both fields.
    $node = Node::create([
      'type' => 'test_content',
      'title' => 'Test Node',
      'field_test_url' => 'https://production.example.com/page',
      'field_another_url' => 'https://production.example.com/other',
    ]);
    $node->save();

    // Create trackers for both fields.
    FieldValueTracker::create([
      'mode' => FieldValueTracker::MODE_REPLACE,
      'target_field' => $this->fieldConfig->id(),
      'prod_value' => 'production.example.com',
      'lower_env_value' => 'staging1.example.com',
    ])->save();

    FieldValueTracker::create([
      'mode' => FieldValueTracker::MODE_REPLACE,
      'target_field' => $another_field->id(),
      'prod_value' => 'production.example.com',
      'lower_env_value' => 'staging2.example.com',
    ])->save();

    // Execute the update command targeting only the first field.
    $this->commands->updateFieldValues($this->fieldConfig->id(), ['dry-run' => FALSE]);

    // Reload node and verify only the targeted field was updated.
    $node = Node::load($node->id());
    $this->assertEquals('https://staging1.example.com/page', $node->get('field_test_url')->value);
    $this->assertEquals('https://production.example.com/other', $node->get('field_another_url')->value);

    // Verify only the targeted field was processed.
    $this->assertTrue(
      $this->testLogger->hasMessage($this->fieldConfig->id()),
      'Targeted field ID appears in log'
    );
    $this->assertFalse(
      $this->testLogger->hasMessage($another_field->id()),
      'Non-targeted field ID does not appear in log'
    );
  }

  /**
   * Tests the update command with non-existent target field.
   */
  public function testUpdateCommandWithNonExistentTarget(): void {
    // Execute the update command with a non-existent target.
    $this->commands->updateFieldValues('node.nonexistent.field_fake', ['dry-run' => FALSE]);

    // Verify warning was logged about no tracker found.
    $this->assertTrue(
      $this->testLogger->hasMessage('No field value tracker found'),
      'Warning logged for non-existent target'
    );
  }

  /**
   * Tests the update command with invalid field target format.
   */
  public function testUpdateCommandWithInvalidFormat(): void {
    // Create a tracker with invalid target format.
    $tracker = FieldValueTracker::create([
      'mode' => FieldValueTracker::MODE_REPLACE,
      'target_field' => $this->fieldConfig->id(),
      'prod_value' => 'production.example.com',
      'lower_env_value' => 'staging.example.com',
    ]);
    $tracker->save();

    // Manually override the target_field to an invalid format.
    // This tests error handling in the command.
    $tracker->set('target_field', 'invalid_format');
    $tracker->save();

    // Execute the update command - should handle error gracefully.
    $this->commands->updateFieldValues(NULL, ['dry-run' => FALSE]);

    // Verify error was logged about invalid format.
    $this->assertTrue(
      $this->testLogger->hasMessage('Invalid field target format'),
      'Error logged for invalid field target format'
    );
  }

  /**
   * Tests that entity storage caches are properly cleared after updates.
   */
  public function testUpdateCommandClearsCaches(): void {
    // Create a test node with a production URL.
    $node = Node::create([
      'type' => 'test_content',
      'title' => 'Test Node',
      'field_test_url' => 'https://production.example.com/page',
    ]);
    $node->save();
    $node_id = $node->id();

    // Load the node to ensure it's in the entity storage cache.
    $cached_node = Node::load($node_id);
    $this->assertEquals('https://production.example.com/page', $cached_node->get('field_test_url')->value);

    // Create a tracker that will update this field.
    $tracker = FieldValueTracker::create([
      'mode' => FieldValueTracker::MODE_REPLACE,
      'target_field' => $this->fieldConfig->id(),
      'prod_value' => 'production.example.com',
      'lower_env_value' => 'staging.example.com',
    ]);
    $tracker->save();

    // Execute the update command (which updates the database directly and
    // should clear entity caches).
    $this->commands->updateFieldValues(NULL, ['dry-run' => FALSE]);

    // Load the node again. If cache clearing works properly, we should see
    // the updated value. If cache clearing didn't work, we'd still see the
    // old cached value.
    $reloaded_node = Node::load($node_id);
    $this->assertEquals(
      'https://staging.example.com/page',
      $reloaded_node->get('field_test_url')->value,
      'Entity cache was properly cleared and node has updated value'
    );

    // Verify success message was logged.
    $this->assertTrue(
      $this->testLogger->hasMessage('Successfully'),
      'Success message logged after update'
    );
  }

  /**
   * Tests production environment safety check for Acquia.
   */
  public function testProductionEnvironmentBlockAcquia(): void {
    // Set Acquia production environment variable.
    putenv('AH_SITE_ENVIRONMENT=prod');

    // Create a tracker.
    FieldValueTracker::create([
      'mode' => FieldValueTracker::MODE_REPLACE,
      'target_field' => $this->fieldConfig->id(),
      'prod_value' => 'production.example.com',
      'lower_env_value' => 'staging.example.com',
    ])->save();

    // Attempt to run the update command.
    $this->commands->updateFieldValues(NULL, ['dry-run' => FALSE]);

    // Verify command was blocked.
    $this->assertTrue(
      $this->testLogger->hasMessage('cannot be run in production'),
      'Error message logged for production environment'
    );
    $this->assertTrue(
      $this->testLogger->hasMessage('lower environments'),
      'Notice message about lower environments logged'
    );

    // Clean up environment variable.
    putenv('AH_SITE_ENVIRONMENT');
  }

  /**
   * Tests production environment safety check for Pantheon.
   */
  public function testProductionEnvironmentBlockPantheon(): void {
    // Set Pantheon production environment variable.
    putenv('PANTHEON_ENVIRONMENT=live');

    // Create a tracker.
    FieldValueTracker::create([
      'mode' => FieldValueTracker::MODE_REPLACE,
      'target_field' => $this->fieldConfig->id(),
      'prod_value' => 'production.example.com',
      'lower_env_value' => 'staging.example.com',
    ])->save();

    // Attempt to run the update command.
    $this->commands->updateFieldValues(NULL, ['dry-run' => FALSE]);

    // Verify command was blocked.
    $this->assertTrue(
      $this->testLogger->hasMessage('cannot be run in production'),
      'Error message logged for production environment'
    );

    // Clean up environment variable.
    putenv('PANTHEON_ENVIRONMENT');
  }

  /**
   * Tests that command runs in non-production environments.
   */
  public function testNonProductionEnvironmentAllowed(): void {
    // Set Acquia staging environment variable.
    putenv('AH_SITE_ENVIRONMENT=test');

    // Create a tracker and node.
    $node = Node::create([
      'type' => 'test_content',
      'title' => 'Test Node',
      'field_test_url' => 'https://production.example.com/page',
    ]);
    $node->save();

    FieldValueTracker::create([
      'mode' => FieldValueTracker::MODE_REPLACE,
      'target_field' => $this->fieldConfig->id(),
      'prod_value' => 'production.example.com',
      'lower_env_value' => 'staging.example.com',
    ])->save();

    // Run the update command.
    $this->commands->updateFieldValues(NULL, ['dry-run' => FALSE]);

    // Verify command ran successfully (not blocked).
    $this->assertFalse(
      $this->testLogger->hasMessage('cannot be run in production'),
      'No production block message in non-production environment'
    );
    $this->assertTrue(
      $this->testLogger->hasMessage('Successfully'),
      'Success message logged in non-production environment'
    );

    // Verify the update actually occurred.
    $node = Node::load($node->id());
    $this->assertEquals('https://staging.example.com/page', $node->get('field_test_url')->value);

    // Clean up environment variable.
    putenv('AH_SITE_ENVIRONMENT');
  }

}
