<?php

namespace Drupal\field_visibility_manager\Form;

use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\user\Entity\Role;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * FieldRoleRestriction Form.
 */
class FieldVisibilityManagerForm extends ConfigFormBase {


  /**
   * The node storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $nodeStorage;
  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * Constructs object.
   *
   * @param \Drupal\Core\Entity\EntityStorageInterface $nodeStorage
   *   The entity storage service.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entityFieldManager
   *   The entity field manager service.
   */
  public function __construct(EntityStorageInterface $nodeStorage, EntityFieldManagerInterface $entityFieldManager) {
    $this->nodeStorage = $nodeStorage;
    $this->entityFieldManager = $entityFieldManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager')->getStorage('node_type'),
      $container->get('entity_field.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'field_visibility_manager_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'field_visibility_manager.adminsettings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $node_types = $this->nodeStorage->loadMultiple();
    // List out all roles.
    $roles = Role::loadMultiple();
    $role_list = [];
    foreach ($roles as $role) {
      $role_list[$role->id()] = $role->label();
    }
    // List of all fields in system.
    foreach ($node_types as $node_type) {
      $entity_type_id = 'node';
      $bundle = $node_type->id();
      $definitions = $this->entityFieldManager->getFieldDefinitions($entity_type_id, $bundle);
      ;
      foreach ($definitions as $key => $val) {
        if (strpos($key, 'field_') !== FALSE) {
          $fieldNames[ucwords($bundle) . " - " . $val->getLabel()] = $key;
        }
      }
    }

    $bundleFields = $fieldNames;
    $config = $this->config('field_visibility_manager.adminsettings');
    // Check configuration is saved.
    if (!$config->get('permissions')) {
      $form['permissions'] = [
        '#type' => 'table',
        '#header' => [$this->t('Field Label'), $this->t('Fields')],
        '#id' => 'permissions',
        '#attributes' => ['class' => ['permissions', 'js-permissions']],
        '#sticky' => TRUE,
      ];
      // Table columns.
      foreach ($role_list as $role) {
        $form['permissions']['#header'][] = [
          'data' => $role,
        ];
      }
      $form['permissions']['#header'][] = [
        'data' => "",
      ];
      // Table row values.
      foreach ($bundleFields as $key => $field) {
        foreach ($role_list as $role) {
          $form['permissions'][$key]['field_label'] = [
            '#type' => 'label',
            '#title' => $key,
          ];
          $form['permissions'][$key]['Field'] = [
            '#type' => 'textfield',
            '#default_value' => $field,
            '#attributes' => ['readonly' => 'readonly'],
          ];
          $form['permissions'][$key][$role] = [
            '#type' => 'checkbox',
            'data' => $field,
          ];
        }
        $form['permissions'][$key]['field_name'] = [
          '#type' => 'value',
          '#value' => $key,
        ];
      }
    }
    $existingarray = [];
    $value = $config->get('permissions') ?: [];
    foreach ($value as $fieldKey => $field) {
      if (is_array($field) && isset($field['Field'])) {
        $existingarray[] = $field['Field'];
      }
    }
    foreach ($definitions as $key => $val) {
      if (strpos($key, 'field_') !== FALSE) {
        $fieldNames[$val->getLabel()] = $key;
      }
    }
    // Newly added fields after configuration save.
    $result = array_diff($bundleFields, $existingarray);
    // Removed fields after configuration save.
    $oldresult = array_diff($existingarray, $bundleFields);
    // Removing the removed fields from listing.
    foreach ($value as $subKey => $subArray) {
      if (is_array($subArray) && isset($subArray['Field'])) {
        foreach ($oldresult as $exvalue) {
          if ($subArray['Field'] == $exvalue) {
            unset($value[$subKey]);
          }
        }
      }
    }
    // Including the newly added fields for listing.
    foreach ($result as $label => $field) {
      $newField = [
        'Field' => $field,
        'field_name' => $label
      ];
      // Initialize all roles to 0
      foreach ($role_list as $role_id => $role_name) {
        $newField[$role_id] = 0;
      }
      $value[$label] = $newField;
    }
    // To bind saved values from config.
    
    $form['permissions'] = [
      '#type' => 'table',
      '#header' => array_merge([$this->t('Field Label'), $this->t('Fields')], array_values($role_list)),
      '#id' => 'permissions',
      '#attributes' => ['class' => ['permissions', 'js-permissions']],
      '#sticky' => TRUE,
    ];
    
    foreach ($value as $key => $field) {
      // Ensure the key is a valid string for render array.
      $row_key = (string) $key;
    
      $form['permissions'][$row_key] = [];
    
      $form['permissions'][$row_key]['field_label'] = [
        '#plain_text' => $field['field_name'],
      ];
    
      $form['permissions'][$row_key]['Field'] = [
        '#type' => 'textfield',
        '#default_value' => $field['Field'],
        '#attributes' => [
          'readonly' => 'readonly',
          'class' => ['checkbox'],
        ],
      ];
    
      foreach ($role_list as $role_id => $role_name) {
        $form['permissions'][$row_key][(string) $role_id] = [
          '#type' => 'checkbox',
          '#default_value' => isset($field[$role_id]) ? $field[$role_id] : 0,
          '#attributes' => ['class' => ['checkbox']],
        ];
      }
    
      $form['permissions'][$row_key]['field_name'] = [
        '#type' => 'hidden',
        '#value' => $field['field_name'],
      ];
    } 
    
    return parent::buildForm($form, $form_state);

  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('field_visibility_manager.adminsettings')
      ->set('permissions', $form_state->getValue('permissions'))
      ->save();
    parent::submitForm($form, $form_state);

  }

}
