<?php

declare(strict_types=1);

namespace Drupal\file_extractor\Extractor;

use Composer\InstalledVersions;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Plugin\DefaultPluginManager;
use Drupal\file_extractor\Attribute\FileExtractorExtractor;

/**
 * Manages FileExtractorExtractor plugins.
 */
class ExtractorPluginManager extends DefaultPluginManager {

  /**
   * {@inheritdoc}
   */
  public function __construct(\Traversable $namespaces, CacheBackendInterface $cache_backend, ModuleHandlerInterface $module_handler) {
    parent::__construct(
      'Plugin/file_extractor/Extractor',
      $namespaces,
      $module_handler,
      ExtractorPluginInterface::class,
      FileExtractorExtractor::class
    );
    $this->alterInfo('file_extractor_extractor_info');
    $this->setCacheBackend($cache_backend, 'file_extractor_extractor_plugins');
  }

  /**
   * {@inheritdoc}
   *
   * @phpstan-ignore-next-line
   */
  protected function alterDefinitions(&$definitions) {
    // Loop through all definitions.
    foreach ($definitions as $definition_key => $definition_info) {
      // Check to see if the dependencies key is set.
      if (is_array($definition_info['moduleDependencies'])) {
        // Loop through dependencies to confirm if enabled.
        foreach ($definition_info['moduleDependencies'] as $dependency) {
          /** @var string $dependency */
          // If dependency is not enabled removed from list of definitions.
          if (!$this->moduleHandler->moduleExists($dependency)) {
            unset($definitions[$definition_key]);
          }
        }
      }

      // Check Composer packages dependencies.
      if (is_array($definition_info['packageDependencies'])) {
        foreach ($definition_info['packageDependencies'] as $package_dependency) {
          /** @var string $package_dependency */
          if (!InstalledVersions::isInstalled($package_dependency)) {
            unset($definitions[$definition_key]);
          }
        }
      }
    }

    parent::alterDefinitions($definitions);
  }

  /**
   * {@inheritdoc}
   */
  public function createInstance($plugin_id, array $configuration = []) {
    /** @var \Drupal\file_extractor\Extractor\ExtractorPluginInterface $instance */
    $instance = parent::createInstance($plugin_id);
    $instance->setConfiguration($configuration);
    return $instance;
  }

}
