<?php

declare(strict_types=1);

namespace Drupal\file_extractor\Service;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\File\FileExists;
use Drupal\Core\File\FileSystemInterface;
use Drupal\file\FileInterface;

/**
 * Provides helper methods for manipulating test files.
 */
class TestFileHelper implements TestFileHelperInterface {

  /**
   * The test file.
   *
   * @var \Drupal\file\FileInterface|null
   */
  protected $testFile;

  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected FileSystemInterface $fileSystem,
    protected ModuleExtensionList $moduleExtensionList,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function getTestFile(): FileInterface {
    if (!isset($this->testFile)) {
      // Copy the source file to the public directory.
      $source = $this->moduleExtensionList->getPath('file_extractor');
      $source .= '/data/' . self::TEST_FILENAME;
      $this->fileSystem->copy($source, self::TEST_FILE_URI, FileExists::Replace);

      /** @var \Drupal\file\FileInterface $file */
      $file = $this->entityTypeManager->getStorage('file')->create([
        'filename' => self::TEST_FILENAME,
        'filemime' => 'application/pdf',
        'uri' => self::TEST_FILE_URI,
        'status' => FileInterface::STATUS_PERMANENT,
      ]);
      $this->testFile = $file;
    }

    return $this->testFile;
  }

  /**
   * {@inheritdoc}
   */
  public function deleteTestFile(): void {
    $this->fileSystem->delete(self::TEST_FILE_URI);
    $this->testFile = NULL;
  }

}
