<?php

declare(strict_types=1);

namespace Drupal\file_mdm_pdf\Plugin\FileMetadata;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\file_mdm\FileMetadataException;
use Drupal\file_mdm\Plugin\Attribute\FileMetadata;
use Drupal\file_mdm\Plugin\FileMetadata\FileMetadataPluginBase;
use \Smalot\PdfParser\Parser;

/**
 * FileMetadata plugin for PDF metadata.
 *
 * Uses the 'PDF parser' library (smalot/pdfparser).
 */
#[FileMetadata(
  id: 'pdf',
  title: new TranslatableMarkup('PDF'),
  help: new TranslatableMarkup('File metadata plugin for PDF metadata.')
)]
class Pdf extends FileMetadataPluginBase {

  private $rawMetadata = NULL;
  private $fileKeys = [];

  /**
   * @return list<string>
   */
  public function getSupportedKeys(?array $options = NULL): array {
    return $this->fileKeys; // array_merge(['Author', 'Title'], $this->fileKeys);
  }

  protected function doGetMetadataFromFile(): mixed {
    $parser = new Parser();
    $file_uri = $this->getLocalTempPath();
    // Get the absolute filesystem path
    $file_system = \Drupal::service('file_system');
    $full_path = $file_system->realpath($file_uri);
    $pdf = $parser->parseFile($full_path);
    $this->rawMetadata = $pdf->getDetails();
    $this->fileKeys = array_keys($this->rawMetadata);
    return $this->rawMetadata;
  }

  /**
   * Validates a file metadata key.
   *
   * @return bool
   *   TRUE if the key is valid.
   *
   * @throws \Drupal\file_mdm\FileMetadataException
   *   In case the key is invalid.
   */
  protected function validateKey(mixed $key, string $method): bool {
    if (!is_string($key)) {
      throw new FileMetadataException("Invalid metadata key specified", $this->getPluginId(), $method);
    }
    if (!in_array($key, $this->getSupportedKeys())) {
      throw new FileMetadataException("Invalid metadata key '{$key}' specified", $this->getPluginId(), $method);
    }
    return TRUE;
  }

  protected function doGetMetadata(mixed $key = NULL): mixed {
    if ($key === NULL) {
      return $this->metadata;
    }
    else {
      $this->validateKey($key, __FUNCTION__);
      if (in_array($key, $this->getSupportedKeys())) {
        return $this->metadata[$key] ?? NULL;
      }
      return NULL;
    }
  }

  protected function doSetMetadata(mixed $key, mixed $value): bool {
    throw new FileMetadataException('Changing PDF metadata is not supported', $this->getPluginId());
  }

  protected function doRemoveMetadata(mixed $key): bool {
    throw new FileMetadataException('Deleting PDF metadata is not supported', $this->getPluginId());
  }
}
