<?php

declare(strict_types=1);

namespace Drupal\Tests\file_mdm_pdf\Kernel;

use Drupal\Core\File\FileExists;
use Drupal\file_mdm\FileMetadataInterface;
use Drupal\file_mdm\FileMetadataManagerInterface;
use Drupal\Tests\file_mdm\Kernel\FileMetadataManagerTestBase;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests that the file metadata 'pdf' plugin works properly.
 */
#[Group('file_mdm')]
class FileMetadataPdfTest extends FileMetadataManagerTestBase {

  protected static $modules = [
    'system',
    'file_mdm',
    'file_mdm_pdf',
    'file_test',
  ];

  public function setUp(): void {
    parent::setUp();
    $this->installConfig(['file_mdm_pdf']);
  }

  /**
   * Tests 'pdf' plugin.
   */
  public function testFontPlugin(): void {
    $this->fileSystem->copy($this->moduleList->getPath('file_mdm_pdf') . '/tests/files/test_metadata.pdf', 'public://', FileExists::Replace);
    // The font files that will be tested.
    $pdf_files = [
      [
        'uri' => 'public://test_metadata.pdf',
        'count_keys' => 7,
        'test_keys' => [
          ['Title', 'Test pdf'],
          ['Author', 'Somatick'],
          ['Subject', 'Test'],
          ['Pages', 1],
        ],
      ],
    ];

    $fmdm = $this->container->get(FileMetadataManagerInterface::class);

    // Walk through test files.
    foreach ($pdf_files as $pdf_file) {
      $file_metadata = $fmdm->uri($pdf_file['uri']);
      if (!$file_metadata) {
        $this->fail("File not found: {$pdf_file['uri']}");
      }
      $this->assertEquals($pdf_file['count_keys'], $this->countMetadataKeys($file_metadata, 'pdf'));
      $this->assertSame(FileMetadataInterface::LOADED_FROM_FILE, $file_metadata->isMetadataLoaded('pdf'));
      foreach ($pdf_file['test_keys'] as $test) {
        $this->assertEquals($test[1], $file_metadata->getMetadata('pdf', $test[0]));
      }
    }
  }

  /**
   * Tests 'pdf' plugin supported keys.
   */
  public function testSupportedKeys(): void {
    $this->fileSystem->copy($this->moduleList->getPath('file_mdm_pdf') . '/tests/files/test_metadata.pdf', 'public://', FileExists::Replace);
    $expected_keys = [
      'Title',
      'Producer',
      'Author',
      'Subject',
      'Keywords',
      'Creator',
      'Pages',
    ];

    $fmdm = $this->container->get(FileMetadataManagerInterface::class);
    $file_md = $fmdm->uri('public://test_metadata.pdf');
    $this->assertEquals($expected_keys, $file_md->getSupportedKeys('pdf'));
  }

}
