<?php

namespace Drupal\file_mime_type_enforcer\Commands;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\file_mime_type_enforcer\Service\MimeTypeValidator;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Queue\QueueFactory;
use Drush\Commands\DrushCommands;
use Symfony\Component\Console\Helper\ProgressBar;

/**
 * Drush commands for auditing file MIME types.
 */
class MimeTypeAuditCommands extends DrushCommands {

  /**
   * Constructs a MimeTypeAuditCommands object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\file_mime_type_enforcer\Service\MimeTypeValidator $mimeTypeValidator
   *   The MIME type validator service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The configuration factory.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   The logger channel factory.
   * @param \Drupal\Core\Queue\QueueFactory $queueFactory
   *   The queue factory service.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected MimeTypeValidator $mimeTypeValidator,
    protected ConfigFactoryInterface $configFactory,
    protected LoggerChannelFactoryInterface $loggerFactory,
    protected QueueFactory $queueFactory,
  ) {
    parent::__construct();
  }

  /**
   * Audit all managed files for MIME type discrepancies.
   *
   * @param array $options
   *   The command options.
   *
   * @option limit
   *   Limit the number of files to process.
   * @option display-errors
   *   Display detailed information only for files with errors or discrepancies.
   * @option reset
   *   Reset the queue and repopulate it with file IDs.
   *
   * @command file-mime-type-enforcer:audit
   * @aliases fmte:audit
   */
  public function auditFiles(
    array $options = [
      'limit' => NULL,
      'display-errors' => FALSE,
      'reset' => FALSE,
    ],
  ) {
    $config = $this->configFactory->get('file_mime_type_enforcer.settings');
    $logger = $this->loggerFactory->get('file_mime_type_enforcer');

    if (!$config->get('enabled')) {
      $this->output()->writeln('<error>File MIME Type Enforcer is not enabled.</error>');
      return;
    }

    // Get the audit queue.
    $queue = $this->queueFactory->get('file_mime_type_audit');

    // Initialize or check queue population.
    $queue_populated = $this->initializeQueue($queue, $options);

    if (!$queue_populated) {
      return;
    }

    $this->output()->writeln('<info>Starting file MIME type audit from queue...</info>');

    // Initialize counters.
    $stats = [
      'processed' => 0,
      'skipped' => 0,
      'matches' => 0,
      'discrepancies' => 0,
      'validation_passed' => 0,
      'validation_failed' => 0,
    ];

    // Get total queue size for progress tracking.
    $total_number_of_items = $queue->numberOfItems();

    $this->output()->writeln("Processing {$total_number_of_items} files from queue...");

    // Initialize progress bar.
    $progress_bar = new ProgressBar($this->output(), $total_number_of_items);
    $progress_bar->setFormat('very_verbose');
    $progress_bar->start();
    $progress_interrupted = FALSE;

    // Process queue items.
    $file_storage = $this->entityTypeManager->getStorage('file');

    while ($queue->numberOfItems() > 0) {
      $item = $queue->claimItem();

      // Guard against NULL items. Break the loop if we fail to claim an item.
      if (!$item) {
        $this->io()->warning('Failed to claim queue item despite item(s) being available. Breaking loop to prevent infinite processing.');
        break;
      }

      try {
        $file = $file_storage->load($item->data);
        if ($file) {
          $this->auditSingleFile($file, $options, $stats, $logger, $progress_bar, $progress_interrupted);
        }
        else {
          $stats['skipped']++;
          if ($options['display-errors']) {
            // Interrupt progress bar for error message.
            if (!$progress_interrupted) {
              $progress_bar->clear();
              $progress_interrupted = TRUE;
            }
            $this->output()->writeln("<error>File ID {$item->data} not found</error>");
          }
        }

      }
      catch (\Exception $e) {
        $stats['skipped']++;
        if ($options['display-errors']) {
          // Interrupt progress bar for error message.
          if (!$progress_interrupted) {
            $progress_bar->clear();
            $progress_interrupted = TRUE;
          }
          $this->output()->writeln("<error>Error processing file ID {$item->data}: {$e->getMessage()}</error>");
        }
      }
      finally {
        // Update progress bar if we processed an item.
        if (is_object($item)) {
          $progress_bar->advance();
        }
        // Delete item from the queue regardless of whether it was processed or
        // not.
        $queue->deleteItem($item);
      }
    }

    // Finish progress bar.
    $progress_bar->finish();

    // Add newline after progress bar (or after error messages).
    $this->output()->writeln('');

    // Display final results.
    $this->displayResults($stats, $total_number_of_items);
  }

  /**
   * Initialize the audit queue with file IDs.
   *
   * @param \Drupal\Core\Queue\QueueInterface $queue
   *   The queue to populate.
   * @param array $options
   *   Command options.
   *
   * @return bool
   *   TRUE if queue is ready for processing, FALSE otherwise.
   */
  protected function initializeQueue($queue, array $options): bool {
    $existing_items = $queue->numberOfItems();

    // If reset option is provided, clear the queue first.
    if ($options['reset']) {
      if ($existing_items > 0) {
        $this->output()->writeln("<info>Flushing {$existing_items} existing queue item(s)... </info>");
        $queue->deleteQueue();
      }
      $existing_items = 0;
    }

    // If queue is empty or was just flushed, populate it.
    if ($existing_items === 0) {
      $this->output()->writeln('<info>Populating queue with file IDs...</info>');

      // Get file storage and build query.
      $file_storage = $this->entityTypeManager->getStorage('file');
      $query = $file_storage->getQuery();
      $query->accessCheck(FALSE);

      if ($options['limit']) {
        $query->range(0, $options['limit']);
      }

      $query->sort('fid', 'ASC');
      $file_ids = $query->execute();
      $total_files = count($file_ids);

      if ($total_files === 0) {
        $this->io()->warning('No files found to audit.');
        return FALSE;
      }

      // Add file IDs to queue with progress bar.
      $queue_progress = new ProgressBar($this->output(), $total_files);
      $queue_progress->setFormat('very_verbose');
      $queue_progress->setMessage('Adding files to queue...', 'message');
      $queue_progress->start();

      $queued_count = 0;
      foreach ($file_ids as $fid) {
        $queue->createItem($fid);
        $queued_count++;
        $queue_progress->advance();
      }

      $queue_progress->finish();
      $this->output()->writeln('');
      $this->output()->writeln("<info>Successfully queued {$queued_count} files for processing.</info>");
    }
    else {
      $this->output()->writeln("<info>Using existing queue with {$existing_items} items. Use --reset to repopulate.</info>");
    }

    return TRUE;
  }

  /**
   * Audit a single file.
   */
  protected function auditSingleFile($file, array $options, array &$stats, $logger, $progress_bar = NULL, &$progress_interrupted = FALSE) {
    try {
      // Run MIME type validation.
      $validation_result = $this->mimeTypeValidator->validateFileMimeType($file);
      $drupal_mime = $validation_result['drupal_mime'];
      $fileinfo_mime = $validation_result['fileinfo_mime'];
      $is_valid = $validation_result['valid'];
      $file_id = $file->id();
      $file_name = $file->getFilename();

      // Check if stored MIME matches what fileinfo detected.
      $matches_expected = ($fileinfo_mime === $drupal_mime);

      if ($matches_expected) {
        $stats['matches']++;
      }
      else {
        $stats['discrepancies']++;
      }

      if ($is_valid) {
        $stats['validation_passed']++;
      }
      else {
        $stats['validation_failed']++;
      }

      // Log results based on module configuration.
      $config = $this->configFactory->get('file_mime_type_enforcer.settings');
      if ($config->get('log_violations') && (!$is_valid || !$matches_expected)) {
        $logger->warning('File audit: @filename (ID: @fid) - Stored: @stored, Expected: @expected, Detected: @detected, Valid: @valid', [
          '@filename' => $file_name,
          '@fid' => $file_id,
          '@expected' => $drupal_mime,
          '@detected' => $fileinfo_mime,
          '@valid' => $is_valid ? 'Yes' : 'No',
        ]);
      }

      // Display errors only when requested and only for files with issues.
      if ($options['display-errors'] && (!$matches_expected || !$is_valid)) {
        // Interrupt progress bar for error message.
        if ($progress_bar && !$progress_interrupted) {
          $progress_bar->clear();
          $progress_interrupted = TRUE;
        }
        $status = $is_valid ? 'PASS' : 'FAIL';
        $match_status = $matches_expected ? 'MATCH' : 'MISMATCH';
        $this->output()->writeln("<error>{$file_name} (ID: {$file_id}) - {$status} / {$match_status}</error>");
        $this->output()->writeln("  Expected: {$drupal_mime}");
        $this->output()->writeln("  Detected: {$fileinfo_mime}");
      }

      // Only increment processed if we successfully completed all validation.
      $stats['processed']++;

    }
    catch (\Exception $e) {
      $stats['skipped']++;
      if ($options['display-errors']) {
        // Interrupt progress bar for error message.
        if ($progress_bar && !$progress_interrupted) {
          $progress_bar->clear();
          $progress_interrupted = TRUE;
        }
        $this->output()->writeln("<error>Skipped {$file_name}: {$e->getMessage()}</error>");
      }
    }
  }

  /**
   * Display final audit results.
   */
  protected function displayResults(array $stats, $total) {
    $this->output()->writeln('');
    $this->output()->writeln('<info>AUDIT COMPLETE</info>');
    $this->output()->writeln("Total Files: {$total}");
    $this->output()->writeln("Processed: {$stats['processed']}");
    $this->output()->writeln("Skipped: {$stats['skipped']}");
    $this->output()->writeln("MIME Matches: {$stats['matches']}");
    $this->output()->writeln("MIME Discrepancies: {$stats['discrepancies']}");
    $this->output()->writeln("Validation Passed: {$stats['validation_passed']}");
    $this->output()->writeln("Validation Failed: {$stats['validation_failed']}");

    if ($stats['validation_failed'] > 0) {
      $this->output()->writeln('<comment>Some files failed validation. Check logs for details.</comment>');
    }
    elseif ($stats['discrepancies'] > 0) {
      $this->output()->writeln('<comment>MIME type discrepancies found but validation passed due to alternative mappings.</comment>');
    }
    else {
      $this->output()->writeln('<info>All files have correct MIME types.</info>');
    }
  }

}
