<?php

namespace Drupal\file_mime_type_enforcer\EventSubscriber;

use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\file_mime_type_enforcer\Service\MimeTypeValidator;
use Drupal\file\Validation\FileValidationEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\Validator\ConstraintViolation;

/**
 * Event subscriber for file validation using FileValidationEvent.
 *
 * It validates file MIME types by comparing results from multiple detection
 * methods and enforces configurable validation rules.
 */
class FileValidationSubscriber implements EventSubscriberInterface {

  use StringTranslationTrait;

  /**
   * Constructs a FileValidationSubscriber object.
   *
   * @param \Drupal\file_mime_type_enforcer\Service\MimeTypeValidator $validator
   *   The MIME type validator service.
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger
   *   The logger channel for this module.
   */
  public function __construct(
    protected MimeTypeValidator $validator,
    protected LoggerChannelInterface $logger,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      FileValidationEvent::class => 'onFileValidate',
    ];
  }

  /**
   * Validates file MIME types during file upload process.
   *
   * This method is called when a FileValidationEvent is dispatched during
   * file uploads. It performs MIME type validation by comparing results
   * from Drupal's ExtensionMimeTypeGuesser with Symfony's fileinfo method.
   *
   * @param \Drupal\file\Validation\FileValidationEvent $event
   *   The file validation event containing the file to validate.
   */
  public function onFileValidate(FileValidationEvent $event): void {
    $file = $event->file;

    try {
      // Perform MIME type validation using the existing service.
      $validation_result = $this->validator->validateFileMimeType($file);

      // If validation fails, add constraint violation.
      if (!$validation_result['valid']) {
        $event->violations->add(new ConstraintViolation(
          $this->t('@message', [
            '@message' => $validation_result['message'],
          ]),
          $validation_result['message'],
          [],
          $file,
          'mimeType',
          $file->getMimeType()
        ));
      }
    }
    catch (\Exception $e) {
      // Log the exception and add a constraint violation for system errors.
      $this->logger->error('Error during MIME type validation for file @filename: @error', [
        '@filename' => $file->getFilename(),
        '@error' => $e->getMessage(),
      ]);

      // In case of service errors, we add a generic validation error
      // but don't completely block the upload unless strict mode is on.
      $config = $this->validator->getConfiguration();
      if ($config['strict_mode']) {
        $event->violations->add(new ConstraintViolation(
          $this->t('Unable to validate file MIME type due to a system error. Please try again or contact an administrator.'),
          'Unable to validate file MIME type due to a system error.',
          [],
          $file,
          'mimeType',
          $file->getMimeType()
        ));
      }
    }
  }

}
