<?php

namespace Drupal\file_mime_type_enforcer\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\file\FileInterface;
use Symfony\Component\Mime\MimeTypeGuesserInterface;
use Symfony\Component\Mime\FileinfoMimeTypeGuesser;

/**
 * Service for validating file MIME types using multiple detection methods.
 *
 * This service compares MIME types detected by Drupal ExtensionMimeTypeGuesser
 * with results from Symfony's fileinfo functionality. It provides configurable
 * alternative MIME type mappings to handle cases where legitimate files might
 * have different but acceptable MIME type signatures.
 */
class MimeTypeValidator {


  /**
   * The Symfony fileinfo MIME type guesser.
   *
   * @var \Symfony\Component\Mime\FileinfoMimeTypeGuesser
   */
  protected FileinfoMimeTypeGuesser $fileinfoGuesser;

  /**
   * Constructor for MimeTypeValidatorService.
   *
   * @param \Symfony\Component\Mime\MimeTypeGuesserInterface $drupalMimeTypeGuesser
   *   The Drupal MIME type guesser service (ExtensionMimeTypeGuesser).
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory service.
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger
   *   The logger channel for this module.
   * @param \Drupal\Core\File\FileSystemInterface $fileSystem
   *   The file system service.
   */
  public function __construct(
    protected MimeTypeGuesserInterface $drupalMimeTypeGuesser,
    protected ConfigFactoryInterface $configFactory,
    protected LoggerChannelInterface $logger,
    protected FileSystemInterface $fileSystem,
  ) {

    // Initialize Symfony's fileinfo guesser.
    $this->fileinfoGuesser = new FileinfoMimeTypeGuesser();
  }

  /**
   * Validates a file's MIME type by comparing multiple detection methods.
   *
   * This method compares the MIME type detected by Drupal's extension-based
   * guesser with the result from Symfony's fileinfo method. If they don't
   * match, it checks configured alternative MIME type mappings for the file
   * extension.
   *
   * @param \Drupal\file\FileInterface $file
   *   The file entity to validate.
   *
   * @return array
   *   An array with validation results containing:
   *   - 'valid': Boolean indicating if the file passes validation
   *   - 'drupal_mime': The MIME type detected by Drupal
   *   - 'fileinfo_mime': The MIME type detected by fileinfo
   *   - 'message': Human-readable message about the validation result
   *   - 'alternative_used': Boolean indicating if alternative mapping was used
   */
  public function validateFileMimeType(FileInterface $file): array {
    $config = $this->configFactory->get('file_mime_type_enforcer.settings');

    // Check if the validator is enabled.
    if (!$config->get('enabled')) {
      return [
        'valid' => TRUE,
        'drupal_mime' => $file->getMimeType(),
        'fileinfo_mime' => NULL,
        'message' => 'MIME type validation is disabled',
        'alternative_used' => FALSE,
      ];
    }

    // Get the file path for fileinfo analysis.
    $file_uri = $file->getFileUri();
    $file_path = $this->fileSystem->realpath($file_uri);

    if (!$file_path || !file_exists($file_path)) {
      $this->logger->warning('File path could not be resolved or file does not exist: @uri', [
        '@uri' => $file_uri,
      ]);
      return [
        'valid' => FALSE,
        'drupal_mime' => $file->getMimeType(),
        'fileinfo_mime' => NULL,
        'message' => 'File path could not be resolved for validation',
        'alternative_used' => FALSE,
      ];
    }

    // Get MIME types from both methods.
    $drupal_mime = $file->getMimeType();
    $fileinfo_mime = $this->fileinfoGuesser->guessMimeType($file_path);

    // If MIME types match exactly, validation passes.
    if ($drupal_mime === $fileinfo_mime) {
      return [
        'valid' => TRUE,
        'drupal_mime' => $drupal_mime,
        'fileinfo_mime' => $fileinfo_mime,
        'message' => 'MIME types match exactly',
        'alternative_used' => FALSE,
      ];
    }

    // MIME types don't match - check for acceptable alternatives.
    $file_extension = strtolower(pathinfo($file->getFilename(), PATHINFO_EXTENSION));
    $mime_type_mappings = $config->get('mime_type_mappings') ?? [];
    $strict_mode = $config->get('strict_mode') ?? FALSE;

    $acceptable_alternative_found = FALSE;

    // Check if the fileinfo-detected MIME type is in acceptable alternatives.
    if (isset($mime_type_mappings[$file_extension])) {
      $acceptable_alternatives = $mime_type_mappings[$file_extension];
      if (in_array($fileinfo_mime, $acceptable_alternatives, TRUE)) {
        $acceptable_alternative_found = TRUE;
      }
    }

    // Determine if validation passes.
    $valid = $acceptable_alternative_found || !$strict_mode;

    // Log violations if configured.
    if ($config->get('log_violations') && (!$valid || $acceptable_alternative_found)) {
      $log_level = $valid ? 'info' : 'warning';
      $this->logger->$log_level('MIME type mismatch for file @filename (@extension): Drupal detected @drupal_mime, fileinfo detected @fileinfo_mime. Alternative mapping used: @alternative', [
        '@filename' => $file->getFilename(),
        '@extension' => $file_extension,
        '@drupal_mime' => $drupal_mime,
        '@fileinfo_mime' => $fileinfo_mime,
        '@alternative' => $acceptable_alternative_found ? 'Yes' : 'No',
      ]);
    }

    $message = $this->buildValidationMessage($valid, $acceptable_alternative_found, $strict_mode, $drupal_mime, $fileinfo_mime);

    return [
      'valid' => $valid,
      'drupal_mime' => $drupal_mime,
      'fileinfo_mime' => $fileinfo_mime,
      'message' => $message,
      'alternative_used' => $acceptable_alternative_found,
    ];
  }

  /**
   * Builds a human-readable validation message.
   *
   * @param bool $valid
   *   Whether the validation passed.
   * @param bool $alternative_used
   *   Whether an alternative MIME type mapping was used.
   * @param bool $strict_mode
   *   Whether strict mode is enabled.
   * @param string $drupal_mime
   *   The Drupal-detected MIME type.
   * @param string $fileinfo_mime
   *   The fileinfo-detected MIME type.
   *
   * @return string
   *   The validation message.
   */
  private function buildValidationMessage(bool $valid, bool $alternative_used, bool $strict_mode, string $drupal_mime, string $fileinfo_mime): string {
    if ($valid && $alternative_used) {
      return "MIME type mismatch detected but acceptable alternative found: Drupal ($drupal_mime) vs Fileinfo ($fileinfo_mime)";
    }

    if ($valid && !$strict_mode) {
      return "MIME type mismatch detected but allowed in non-strict mode: Drupal ($drupal_mime) vs Fileinfo ($fileinfo_mime)";
    }

    if (!$valid) {
      return "MIME type validation failed: Drupal detected ($drupal_mime) but fileinfo detected ($fileinfo_mime) with no acceptable alternative mapping.";
    }

    return "MIME type validation passed";
  }

  /**
   * Gets the current configuration for the MIME type validator.
   *
   * @return array
   *   The configuration array.
   */
  public function getConfiguration(): array {
    $config = $this->configFactory->get('file_mime_type_enforcer.settings');
    return [
      'enabled' => $config->get('enabled'),
      'strict_mode' => $config->get('strict_mode'),
      'log_violations' => $config->get('log_violations'),
      'mime_type_mappings' => $config->get('mime_type_mappings'),
    ];
  }

}
