<?php

namespace Drupal\file_mime_type_enforcer\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Extension\ExtensionPathResolver;
use Drupal\Component\Serialization\Yaml;

/**
 * Configuration form for the File MIME Type Enforcer module.
 *
 * This form allows administrators to configure MIME type validation settings
 * including alternative MIME type mappings for file extensions.
 */
class MimeTypeEnforcerSettingsForm extends ConfigFormBase {

  /**
   * Constructs a MimeTypeEnforcerSettingsForm object.
   *
   * @param \Drupal\Core\Extension\ExtensionPathResolver $extensionPathResolver
   *   The extension path resolver service.
   * @param \Drupal\Component\Serialization\Yaml $yamlSerialization
   *   The YAML serialization service.
   */
  public function __construct(
    protected ExtensionPathResolver $extensionPathResolver,
    protected Yaml $yamlSerialization,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('extension.path.resolver'),
      $container->get('serialization.yaml'),
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['file_mime_type_enforcer.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'file_mime_type_enforcer_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('file_mime_type_enforcer.settings');

    $form['general'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('General Settings'),
      '#collapsible' => FALSE,
    ];

    $form['general']['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable MIME type enforcement'),
      '#description' => $this->t("When enabled, file uploads will be validated by comparing Drupal's ExtensionMimeTypeGuesser with Symfony's fileinfo method."),
      '#default_value' => $config->get('enabled') ?? TRUE,
    ];

    $form['general']['strict_mode'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Strict mode'),
      '#description' => $this->t("When enabled, files will be rejected if MIME types don't match, even if no alternative mapping is configured. When disabled, mismatched MIME types will be logged but allowed."),
      '#default_value' => $config->get('strict_mode') ?? FALSE,
    ];

    $form['general']['log_violations'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Log MIME type violations'),
      '#description' => $this->t('When enabled, MIME type mismatches will be logged for monitoring purposes.'),
      '#default_value' => $config->get('log_violations') ?? TRUE,
    ];

    $form['mappings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('MIME Type Alternative Mappings'),
      '#description' => $this->t('Define acceptable alternative MIME types for specific file extensions. This is useful when legitimate files might have different but acceptable MIME type signatures detected by different methods.'),
      '#collapsible' => TRUE,
      '#collapsed' => FALSE,
    ];

    // Convert current mappings to JSON for display.
    $current_mappings = $config->get('mime_type_mappings') ?? [];
    $mappings_json = !empty($current_mappings) ? json_encode($current_mappings, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES) : '';

    $form['mappings']['mime_type_mappings'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Alternative MIME type mappings (JSON format)'),
      '#description' => $this->t('Define alternative MIME types for specific file extensions in JSON format. <br><strong>Example:</strong><br><pre>{
  "jpg": ["image/jpeg", "image/pjpeg"],
  "pdf": ["application/pdf", "application/x-pdf"],
  "docx": ["application/vnd.openxmlformats-officedocument.wordprocessingml.document", "application/zip"]
}</pre>'),
      '#default_value' => $mappings_json,
      '#rows' => 15,
      '#attributes' => [
        'style' => 'font-family: monospace; font-size: 12px;',
      ],
    ];

    $form['mappings']['reset_defaults'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Reset to default mappings'),
      '#description' => $this->t('Check this box to reset the MIME type mappings to the default values provided by the module.'),
      '#default_value' => FALSE,
    ];

    // Add examples section.
    $form['examples'] = [
      '#type' => 'details',
      '#title' => $this->t('Common MIME Type Issues & Solutions'),
      '#collapsed' => TRUE,
    ];

    $examples = [
      'JPEG images' => 'Some JPEG files may be detected as "image/pjpeg" by fileinfo while Drupal expects "image/jpeg"',
      'Office documents' => 'Modern Office documents (.docx, .xlsx) may be detected as "application/zip" since they are ZIP archives',
      'PDF files' => 'PDF files might be detected as "application/x-pdf" instead of "application/pdf"',
      'Text files' => 'Plain text files may vary between "text/plain" and "text/x-plain"',
    ];

    $examples_markup = '<ul>';
    foreach ($examples as $case => $description) {
      $examples_markup .= '<li><strong>' . $case . ':</strong> ' . $description . '</li>';
    }
    $examples_markup .= '</ul>';

    $form['examples']['info'] = [
      '#markup' => $examples_markup,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $mappings_input = $form_state->getValue('mime_type_mappings');

    // Skip validation if reset to defaults is checked.
    if ($form_state->getValue('reset_defaults')) {
      return;
    }

    // Validate JSON format if mappings are provided.
    if (!empty($mappings_input)) {
      $decoded_mappings = json_decode($mappings_input, TRUE);

      if (json_last_error() !== JSON_ERROR_NONE) {
        $form_state->setErrorByName('mime_type_mappings', $this->t('Invalid JSON format. Please check your syntax. Error: @error', [
          '@error' => json_last_error_msg(),
        ]));
        return;
      }

      // Validate structure - should be extension => array of MIME types.
      if (!is_array($decoded_mappings)) {
        $form_state->setErrorByName('mime_type_mappings', $this->t('Mappings must be a JSON object with file extensions as keys and arrays of MIME types as values.'));
        return;
      }

      foreach ($decoded_mappings as $extension => $mime_types) {
        if (!is_string($extension)) {
          $form_state->setErrorByName('mime_type_mappings', $this->t('All extension keys must be strings.'));
          return;
        }

        if (!is_array($mime_types)) {
          $form_state->setErrorByName('mime_type_mappings', $this->t('All MIME type values must be arrays for extension "@extension".', [
            '@extension' => $extension,
          ]));
          return;
        }

        foreach ($mime_types as $mime_type) {
          if (!is_string($mime_type) || !preg_match('/^[a-z-]+\/[a-z0-9\.\+\-]+$/i', $mime_type)) {
            $form_state->setErrorByName('mime_type_mappings', $this->t('Invalid MIME type "@mime_type" for extension "@extension". MIME types should follow the format "type/subtype".', [
              '@mime_type' => $mime_type,
              '@extension' => $extension,
            ]));
            return;
          }
        }
      }
    }

    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $config = $this->config('file_mime_type_enforcer.settings');

    // Handle reset to defaults.
    if ($form_state->getValue('reset_defaults')) {
      // Load default mappings from the installed config.
      $module_path = $this->extensionPathResolver->getPath('module', 'file_mime_type_enforcer');
      $install_config_path = $module_path . '/config/install/file_mime_type_enforcer.settings.yml';

      // If the file exists, read the default mappings from the file.
      if (file_exists($install_config_path)) {
        $default_config = $this->yamlSerialization->decode(file_get_contents($install_config_path));
        $default_mappings = $default_config['mime_type_mappings'] ?? [];
        $config->set('mime_type_mappings', $default_mappings);
        $this->messenger()->addStatus($this->t('MIME type mappings have been reset to default values.'));
      }
      else {
        $this->messenger()->addError($this->t('Could not find default configuration file. Reset failed.'));
      }
    }
    else {
      // Process the JSON mappings.
      $mappings_input = $form_state->getValue('mime_type_mappings');
      $mappings = !empty($mappings_input) ? json_decode($mappings_input, TRUE) : [];
      $config->set('mime_type_mappings', $mappings);
    }

    // Save other settings.
    $config
      ->set('enabled', $form_state->getValue('enabled'))
      ->set('strict_mode', $form_state->getValue('strict_mode'))
      ->set('log_violations', $form_state->getValue('log_violations'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
