<?php

namespace Drupal\file_resup\Form;

use Drupal\Component\Utility\Bytes;
use Drupal\Component\Utility\Environment;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\StringTranslation\ByteSizeMarkup;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Implements hook_form_FORM_ID_alter().
 */
class FieldConfigFormAlter implements ContainerInjectionInterface {

  use StringTranslationTrait;

  private EntityTypeManagerInterface $entityTypeManager;

  public function __construct(EntityTypeManagerInterface $entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
  }

  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager')
    );
  }

  /**
   * Implements hook_form_FORM_ID_alter().
   */
  public function formAlter(&$form, FormStateInterface $form_state) {
    $field_config = $form_state->getFormObject()->getEntity();
    $settings = $field_config->getFieldStorageDefinition()->getSettings();
    $third_party_settings = $field_config->getThirdPartySettings('file_resup') ?? [];
    // Is this a file field?
    if (!empty($settings['target_type']) && $settings['target_type'] === 'file') {
      $form['resumable_upload'] = [
        '#type' => 'details',
        '#title' => $this->t('Resumable Upload Settings'),
        '#open' => $third_party_settings['enabled'] ?? FALSE,
        '#tree' => TRUE,
        'settings' => [
          'enabled' => [
            '#type' => 'checkbox',
            '#title' => $this->t('Enable resumable upload'),
            '#default_value' => $third_party_settings['enabled'] ?? FALSE,
          ],
          'max_upload_size' => [
            '#type' => 'textfield',
            '#title' => $this->t('Maximum upload size'),
            '#default_value' => $third_party_settings['max_upload_size'] ?? NULL,
            '#description' => $this->t('Enter a value like "512" (bytes), "80 KB" (kilobytes) or "50 MB" (megabytes) in order to restrict the allowed file size. If left empty the file sizes could be limited only by PHP\'s maximum post and file upload sizes (current limit <strong>%limit</strong>).', [
              '%limit' => ByteSizeMarkup::create(Environment::getUploadMaxSize()),
            ]),
            '#size' => 10,
            '#element_validate' => [[static::class, 'validateMaxUploadSize']],
          ],
          'auto_upload' => [
            '#type' => 'checkbox',
            '#title' => $this->t('Start upload on files added'),
            '#description' => $this->t('When checked, upload will start as soon as files are added without requiring to click Upload, unless some of the added files did not pass validation.'),
            '#default_value' => $third_party_settings['auto_upload'] ?? NULL,
          ],
        ],
      ];
      $form['actions']['submit']['#submit'][] = [$this, 'submit'];
    }
  }

  /**
   * @see \Drupal\file\Plugin\Field\FieldType\FileItem::validateMaxFilesize
   */
  public static function validateMaxUploadSize($element, FormStateInterface $form_state) {
    $element['#value'] = trim($element['#value']);
    $form_state->setValue(['settings', 'max_upload_size'], $element['#value']);
    if (!empty($element['#value']) && !Bytes::validate($element['#value'])) {
      $form_state->setError($element, new TranslatableMarkup('The "@name" option must contain a valid value. You may either leave the text field empty or enter a string like "512" (bytes), "80 KB" (kilobytes) or "50 MB" (megabytes).', ['@name' => $element['#title']]));
    }
  }

  public function submit(&$form, FormStateInterface $form_state) {
    $field_config = $form_state->getFormObject()->getEntity();
    $settings = $form_state->getValue('resumable_upload')['settings'];
    $field_config->setThirdPartySetting('file_resup', 'enabled', $settings['enabled']);
    $field_config->setThirdPartySetting('file_resup', 'max_upload_size', $settings['max_upload_size']);
    $field_config->setThirdPartySetting('file_resup', 'auto_upload', $settings['auto_upload']);
    $field_config->save();
  }

}
