<?php

namespace Drupal\Tests\file_upload_secure_validator\Kernel;

use Drupal\file\Entity\File;
use Drupal\KernelTests\KernelTestBase;
use PHPUnit\Framework\Attributes\RunTestsInSeparateProcesses;

/**
 * Tests the file_upload_secure_validator service integration.
 *
 * @group file_upload_secure_validator
 */
#[RunTestsInSeparateProcesses]
class ServiceIntegrationTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'file',
    'system',
    'user',
    'file_upload_secure_validator',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installEntitySchema('user');
    $this->installEntitySchema('file');
    $this->installSchema('file', ['file_usage']);
    $this->installConfig(['file_upload_secure_validator']);
  }

  /**
   * Tests that the service is properly registered.
   */
  public function testServiceIsRegistered() {
    $service = \Drupal::service('file_upload_secure_validator');
    $this->assertNotNull($service, 'Service is registered and available');
    $this->assertInstanceOf(
      'Drupal\file_upload_secure_validator\Service\FileUploadSecureValidator',
      $service,
      'Service is an instance of FileUploadSecureValidator'
    );
  }

  /**
   * Tests that the service can be dependency injected.
   */
  public function testServiceDependencyInjection() {
    $container = \Drupal::getContainer();
    
    // Test that the service can be retrieved from the container.
    $this->assertTrue($container->has('file_upload_secure_validator'), 'Service is in the container');
    
    $service = $container->get('file_upload_secure_validator');
    $this->assertNotNull($service, 'Service can be retrieved from container');
  }

  /**
   * Tests that the service validates files correctly.
   */
  public function testServiceValidatesFiles() {
    $service = \Drupal::service('file_upload_secure_validator');
    
    // Get the module path for test resources.
    $module_path = \Drupal::service('extension.list.module')->getPath('file_upload_secure_validator');
    $test_file_path = \Drupal::root() . '/' . $module_path . '/tests/src/Unit/resources/original_pdf.pdf';
    
    // Create a file entity.
    $file = File::create([
      'uri' => $test_file_path,
      'filename' => 'test.pdf',
      'filemime' => 'application/pdf',
    ]);
    
    // Validate the file.
    $errors = $service->validate($file);
    
    // Should have no errors for a legitimate PDF.
    $this->assertIsArray($errors, 'Validation returns an array');
    $this->assertEmpty($errors, 'No errors for legitimate PDF file');
  }

  /**
   * Tests that the service uses configuration.
   */
  public function testServiceUsesConfiguration() {
    $service = \Drupal::service('file_upload_secure_validator');
    $config = \Drupal::configFactory()->getEditable('file_upload_secure_validator.settings');
    
    // Get the module path for test resources.
    $module_path = \Drupal::service('extension.list.module')->getPath('file_upload_secure_validator');
    
    // Test with a CSV file (which is in the equivalence groups).
    $csv_file_path = \Drupal::root() . '/' . $module_path . '/tests/src/Unit/resources/original_csv.csv';
    $csv_file = File::create([
      'uri' => $csv_file_path,
      'filename' => 'test.csv',
      'filemime' => 'text/csv',
    ]);
    
    $errors = $service->validate($csv_file);
    $this->assertEmpty($errors, 'CSV file passes validation with default config');
    
    // Modify the configuration to remove CSV equivalence groups.
    $config->set('mime_types_equivalence_groups', [])->save();
    
    // CSV should still pass because the MIME types match exactly.
    $errors = $service->validate($csv_file);
    $this->assertEmpty($errors, 'CSV file still passes when MIME types match exactly');
  }

  /**
   * Tests that the service detects MIME type mismatches.
   */
  public function testServiceDetectsMismatch() {
    $service = \Drupal::service('file_upload_secure_validator');
    
    // Get the module path for test resources.
    $module_path = \Drupal::service('extension.list.module')->getPath('file_upload_secure_validator');
    
    // Test with a falsified file (PDF with .txt extension).
    $falsified_file_path = \Drupal::root() . '/' . $module_path . '/tests/src/Unit/resources/original_pdf.txt';
    $falsified_file = File::create([
      'uri' => $falsified_file_path,
      'filename' => 'test.txt',
      'filemime' => 'text/plain',
    ]);
    
    $errors = $service->validate($falsified_file);
    
    $this->assertNotEmpty($errors, 'Falsified file generates errors');
    $this->assertCount(1, $errors, 'Exactly one error is returned');
    
    // Check the error message.
    $error_message = (string) $errors[0];
    $this->assertStringContainsString('problem with this file', $error_message);
  }

  /**
   * Tests that the service handles missing files gracefully.
   */
  public function testServiceHandlesMissingFiles() {
    $service = \Drupal::service('file_upload_secure_validator');
    
    // Create a file object pointing to a non-existent file.
    $missing_file = File::create([
      'uri' => '/tmp/nonexistent_file_test.pdf',
      'filename' => 'missing.pdf',
      'filemime' => 'application/pdf',
    ]);
    
    $errors = $service->validate($missing_file);
    
    $this->assertNotEmpty($errors, 'Missing file generates errors');
    $this->assertCount(1, $errors, 'Exactly one error is returned');
    
    // Check that the error mentions the exception.
    $error_message = (string) $errors[0];
    $this->assertStringContainsString('InvalidArgumentException', $error_message);
  }

  /**
   * Tests that the service respects MIME type equivalence groups.
   */
  public function testServiceRespectsEquivalenceGroups() {
    $service = \Drupal::service('file_upload_secure_validator');
    
    // Get the module path for test resources.
    $module_path = \Drupal::service('extension.list.module')->getPath('file_upload_secure_validator');
    
    // Test with an XML file.
    // According to the default config, text/xml and application/xml are equivalent.
    $xml_file_path = \Drupal::root() . '/' . $module_path . '/tests/src/Unit/resources/original_xml.xml';
    
    // Create a file with text/xml MIME type.
    $xml_file = File::create([
      'uri' => $xml_file_path,
      'filename' => 'test.xml',
      'filemime' => 'text/xml',
    ]);
    
    $errors = $service->validate($xml_file);
    $this->assertEmpty($errors, 'XML file passes validation with text/xml MIME type');
  }

}
