<?php

declare(strict_types=1);

namespace Drupal\file_visibility_track_usage\Hook;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Hook\Attribute\ReorderHook;
use Drupal\Core\Hook\Order\OrderAfter;
use Drupal\file_visibility\FileVisibilityPluginManager;
use Drupal\file_visibility\Hook\EntityOperationHooks;
use Drupal\track_usage\Entity\TrackConfigInterface;

/**
 * File Visibility Track Usage hooks.
 */
class FileVisibilityTrackUsageHook {

  public function __construct(
    protected readonly ConfigFactoryInterface $configFactory,
    protected readonly FileVisibilityPluginManager $pluginManager,
  ) {}

  /**
   * Clears FileVisibilityPluginManager cache if track usage config is changing.
   */
  #[Hook('track_usage_config_update')]
  #[Hook('track_usage_config_delete')]
  public function clearFileVisibilityPluginManagerCache(TrackConfigInterface $config): void {
    $configId = $this->configFactory->get('file_visibility_track_usage.settings')
      ->get('track_usage_config');

    if ($configId === $config->id()) {
      $this->pluginManager->clearCachedDefinitions();
    }
  }

  /**
   * Ensures that hooks from Track Usages module run first.
   */
  #[ReOrderHook(
    hook: 'entity_insert',
    class: EntityOperationHooks::class,
    method: 'updateFileVisibility',
    order: new OrderAfter(['track_usage'])
  )]
  #[ReOrderHook(
    hook: 'entity_update',
    class: EntityOperationHooks::class,
    method: 'updateFileVisibility',
    order: new OrderAfter(['track_usage'])
  )]
  #[ReOrderHook(
    hook: 'entity_delete',
    class: EntityOperationHooks::class,
    method: 'updateFileVisibility',
    order: new OrderAfter(['track_usage'])
  )]
  #[ReOrderHook(
    hook: 'entity_translation_delete',
    class: EntityOperationHooks::class,
    method: 'updateFileVisibility',
    order: new OrderAfter(['track_usage'])
  )]
  #[ReOrderHook(
    hook: 'entity_revision_delete',
    class: EntityOperationHooks::class,
    method: 'updateFileVisibility',
    order: new OrderAfter(['track_usage'])
  )]
  public function reorderHooks(): void {}

}
