<?php

declare(strict_types=1);

namespace Drupal\file_visibility_track_usage\Plugin\FileVisibility;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ConfigInstallerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\file\FileInterface;
use Drupal\file_visibility\Attribute\FileVisibility;
use Drupal\file_visibility\FileVisibilityPluginBase;
use Drupal\track_usage\Entity\TrackConfigInterface;
use Drupal\track_usage\Recorder;
use Drupal\track_usage\RecorderInterface;
use Drupal\track_usage\Trait\EntityUtilityTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Detects relations between source entities and files.
 *
 * The plugin is using usage records created by the Track Usages module. Track
 * Usages module knows how to track the relation between a source entity, such
 * as nodes, and files, by traversing the traversable (middle-level) entities
 * (media, paragraph, etc.), and reaching the file entities.
 */
#[FileVisibility(
  id: 'track_usage',
  label: new TranslatableMarkup('Track Usages'),
  description: new TranslatableMarkup('Uses the Track Usages module to get the relation between source entities and files'),
)]
class TrackUsage extends FileVisibilityPluginBase {

  use EntityUtilityTrait;

  /**
   * Track Usages configuration to be used by this plugin.
   */
  protected TrackConfigInterface $trackConfig;

  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    protected readonly ConfigFactoryInterface $configFactory,
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly RecorderInterface $recorder,
    protected readonly ConfigInstallerInterface $configInstaller,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('config.factory'),
      $container->get('entity_type.manager'),
      $container->get(Recorder::class),
      $container->get('config.installer'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function applies(): bool {
    // The configuration is set and exists.
    return $this->getTrackConfig()
      // At least one source entity type has been set.
      && !empty($this->getTrackConfig()->get('source'))
      // The 'file' entity type is in the target list.
      && array_key_exists('file', $this->getTrackConfig()->get('target'))
      // Is configured to record entity changes in real time.
      && $this->getTrackConfig()->useRealTimeRecording();
  }

  /**
   * {@inheritdoc}
   */
  public function getEntityFiles(FieldableEntityInterface $entity): iterable {
    foreach ($this->recorder->getTargetsForEntity($entity, 'file', $this->getTrackConfig()) as $fid) {
      yield $fid;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getPathsToFile(FileInterface $file): iterable {
    foreach ($this->recorder->getPathsToTarget($file, $this->getTrackConfig()) as $path) {
      foreach ($path as $delta => [$entityTypeId, $entityId, $revisionId]) {
        // Replaces the entity key with the entity object.
        $path[$delta] = $this->loadEntity($entityTypeId, $entityId, $revisionId);
      }
      yield $path;
    }
  }

  /**
   * Returns and caches the Track Usage configuration.
   *
   * @return \Drupal\track_usage\Entity\TrackConfigInterface|null
   *   The Track Usage configuration or NULL.
   */
  protected function getTrackConfig(): ?TrackConfigInterface {
    if ($this->configInstaller->isSyncing()) {
      return NULL;
    }

    if (!isset($this->trackConfig)) {
      $configId = $this->configFactory
        ->get('file_visibility_track_usage.settings')
        ->get('track_usage_config');

      if (!$configId || !is_string($configId)) {
        return NULL;
      }

      $this->trackConfig = $this->entityTypeManager
        ->getStorage('track_usage_config')
        ->load($configId);
    }

    return $this->trackConfig;
  }

}
