<?php

declare(strict_types=1);

namespace Drupal\file_visibility;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Session\AnonymousUserSession;
use Drupal\Core\Site\Settings;
use Drupal\Core\StreamWrapper\StreamWrapperManager;
use Drupal\file\FileInterface;
use Drupal\file\FileRepositoryInterface;
use Drupal\track_usage\Entity\TrackConfigInterface;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Default implementation for the module's main service.
 */
class FileVisibility implements FileVisibilityInterface {

  private const CORE_SCHEMES = ['public', 'private', 'temporary'];

  /**
   * Public schemes.
   *
   * @var string[]
   */
  protected array $publicSchemes;

  /**
   * The entities to be processed.
   *
   * @var list<\Drupal\Core\Entity\FieldableEntityInterface>
   */
  protected array $entities = [];

  /**
   * Static cache of track usage config.
   */
  protected TrackConfigInterface $trackConfig;

  public function __construct(
    protected readonly FileVisibilityPluginManager $manager,
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly FileSystemInterface $fileSystem,
    protected readonly FileRepositoryInterface $fileRepository,
    protected readonly AccountInterface $currentUser,
    protected readonly ConfigFactoryInterface $configFactory,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [KernelEvents::RESPONSE => 'doUpdateFileVisibility'];
  }

  /**
   * {@inheritdoc}
   */
  public function updateFilesVisibility(EntityInterface $entity): void {
    // Prevent any operation if no plugins are available.
    if (!$this->manager->getPlugins()) {
      return;
    }

    if ($entity instanceof FieldableEntityInterface && !$entity instanceof FileInterface) {
      // The entity is limited to types that can determine the file usages.
      // Source entities are included (e.g., node), but also traversable
      // entities, such as media or paragraph; but not files.
      $this->entities[] = $entity;
    }
  }

  /**
   * Processes file visibility once a response was created.
   */
  public function doUpdateFileVisibility(): void {
    if (!$this->entities) {
      return;
    }

    $anonymous = new AnonymousUserSession();

    while ($this->entities) {
      $entity = array_shift($this->entities);

      // Make sure the access control handler static cache is not stale.
      $this->entityTypeManager->getAccessControlHandler($entity->getEntityTypeId())->resetCache();

      // Other processes might have changed the files used by this source
      // entity, so we need to re-check the files. Merge the initial files with
      // the current ones because some files might have been added and others
      // removed.
      $fids = $this->getEntityUsedFiles($entity);
      if (!$fids) {
        continue;
      }

      $fileStorage = $this->entityTypeManager->getStorage('file');
      foreach ($fileStorage->loadMultiple($fids) as $file) {
        if (!file_exists($file->getFileUri()) || $file->isTemporary()) {
          // The file might have been deleted or is temporary.
          continue;
        }

        $access = $this->getFileAccess($file, $anonymous);

        // No path to the file is accessible, and the file is public.
        if ($access->isForbidden() && $this->hasPublicScheme($file->getFileUri())) {
          $this->moveFile($file, static::PRIVATE_LOCATION . StreamWrapperManager::getTarget($file->getFileUri()));
        }
        // At least one path to the file is accessible, and the file is private.
        elseif ($access->isAllowed() && str_starts_with($file->getFileUri(), static::PRIVATE_LOCATION)) {
          [, $path] = explode('/', StreamWrapperManager::getTarget($file->getFileUri()), 2);
          $this->moveFile($file, $this->getOriginalPublicScheme($file) . "://$path");
        }
        // There are no paths to file. Restore the file location, if case.
        elseif ($access->isNeutral() && !$file->get('originalUri')->isEmpty() && $file->getFileUri() !== $file->get('originalUri')) {
          $this->moveFile($file, $file->get('originalUri'));
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getEntityUsedFiles(FieldableEntityInterface $entity): array {
    $fids = [];

    // Collect files from plugins.
    foreach ($this->manager->getPlugins() as $plugin) {
      foreach ($plugin->getEntityFiles($entity) as $fid) {
        $fids[$fid] = $fid;
      }
    }

    return array_values($fids);
  }

  /**
   * {@inheritdoc}
   */
  public function getFileAccess(FileInterface $file, AccountInterface $account): AccessResultInterface {
    $paths = [];
    foreach ($this->manager->getPlugins() as $plugin) {
      foreach ($plugin->getPathsToFile($file) as $path) {
        $paths[] = $path;
      }
    }

    $metadata = (new CacheableMetadata())->addCacheableDependency($file);

    if (!$paths) {
      // There are no paths to this file. We can't make an opinion.
      return AccessResult::neutral('No paths to file')->addCacheableDependency($metadata);
    }

    foreach ($paths as $path) {
      $pathIsAccessible = NULL;

      foreach ($path as $entity) {
        if ($entity === NULL) {
          continue;
        }
        assert($entity instanceof FieldableEntityInterface);

        // Avoid needless calls to $this->getEntityAccess(...).
        if ($pathIsAccessible !== FALSE) {
          $entityAccess = $this->getEntityAccess($entity, $account);
          $pathIsAccessible = $entityAccess->isAllowed();
          $metadata->addCacheableDependency($entityAccess);
        }
        $metadata->addCacheableDependency($entity);
      }

      // This path allows $account to reach the file. We're done.
      if ($pathIsAccessible) {
        return AccessResult::allowed()->addCacheableDependency($metadata);
      }
    }

    // No path is accessible for $account.
    return AccessResult::forbidden('No accessible path')->addCacheableDependency($metadata);
  }

  /**
   * Moves a given file to a given destination.
   *
   * @param \Drupal\file\FileInterface $file
   *   The file entity.
   * @param string $destinationUri
   *   The destination URI.
   */
  protected function moveFile(FileInterface $file, string $destinationUri): void {
    $destinationDir = $this->fileSystem->dirname($destinationUri);
    if (!$this->fileSystem->prepareDirectory($destinationDir, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS)) {
      throw new \RuntimeException("Directory $destinationDir cannot be created or it's not writable.");
    }
    $this->fileRepository->move($file, $destinationUri);
  }

  /**
   * Returns the public schemes.
   *
   * @return string[]
   *   Public schemes.
   */
  protected function getPublicSchemes(): array {
    if (!isset($this->publicSchemes)) {
      $additionalPublicSchemes = array_diff(Settings::get('file_additional_public_schemes', []), self::CORE_SCHEMES);
      $this->publicSchemes = ['public', ...$additionalPublicSchemes];
    }
    return $this->publicSchemes;
  }

  /**
   * Checks whether the URI has a public scheme.
   *
   * @param string $uri
   *   The file URI.
   *
   * @return bool
   *   Whether the URI has a public scheme.
   */
  protected function hasPublicScheme(string $uri): bool {
    $scheme = StreamWrapperManager::getScheme($uri);
    return in_array($scheme, $this->getPublicSchemes(), TRUE);
  }

  /**
   * Gets the original scheme if it's a public scheme, or fallback to 'public'.
   *
   * @param \Drupal\file\FileInterface $file
   *   The file entity.
   *
   * @return string
   *   The original scheme if it's a public scheme, or fallback to 'public'.
   */
  protected function getOriginalPublicScheme(FileInterface $file): string {
    if ($this->hasPublicScheme($file->get('originalUri')->value)) {
      return StreamWrapperManager::getScheme($file->get('originalUri')->value);
    }
    // The original URI doesn't have a public scheme, use 'public'.
    return 'public';
  }

  /**
   * Computes the access to entity.
   *
   * If the entity type is exposing a canonical URL, the access to the canonical
   * URL takes precedence.
   *
   * @param \Drupal\Core\Entity\FieldableEntityInterface $entity
   *   The entity.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The user account.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result object.
   */
  protected function getEntityAccess(FieldableEntityInterface $entity, AccountInterface $account): AccessResultInterface {
    if ($this->getTrackUsageConfig()->isSource($entity) && $entity->getEntityType()->hasLinkTemplate('canonical')) {
      return $entity->toUrl()->access($account, TRUE);
    }
    else {
      return $entity->access('view', $account, TRUE);
    }
  }

  /**
   * Returns the Track Usage config used in file visibility.
   *
   * @return \Drupal\track_usage\Entity\TrackConfigInterface
   *   The track usage config entity.
   */
  protected function getTrackUsageConfig(): TrackConfigInterface {
    if (!isset($this->trackUsageConfig)) {
      $configId = $this->configFactory->get('file_visibility_track_usage.settings')
        ->get('track_usage_config');
      $this->trackConfig = $this->entityTypeManager->getStorage('track_usage_config')
        ->load($configId);
    }
    return $this->trackConfig;
  }

}
