<?php

declare(strict_types=1);

namespace Drupal\file_visibility;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Plugin\DefaultPluginManager;
use Drupal\file_visibility\Attribute\FileVisibility;

/**
 * Plugin manager for FileVisibility plugins.
 */
class FileVisibilityPluginManager extends DefaultPluginManager {

  /**
   * Static cache plugin instances.
   *
   * @var array<array-key, \Drupal\file_visibility\FileVisibilityPluginInterface>|null
   */
  protected ?array $instances;

  public function __construct(
    \Traversable $namespaces,
    CacheBackendInterface $cacheBackend,
    ModuleHandlerInterface $moduleHandler,
    protected readonly ConfigFactoryInterface $configFactory,
  ) {
    parent::__construct('Plugin/FileVisibility', $namespaces, $moduleHandler, FileVisibilityPluginInterface::class, FileVisibility::class);
    $this->alterInfo('file_visibility_info');
    $this->setCacheBackend($cacheBackend, 'file_visibility_plugins');
  }

  /**
   * Returns the enabled and applicable plugin instances.
   *
   * @return array<array-key, \Drupal\file_visibility\FileVisibilityPluginInterface>
   *   Enabled plugin definitions.
   */
  public function getPlugins(): array {
    if (!isset($this->instances)) {
      $enabledPlugins = $this->configFactory->get('file_visibility.settings')->get('enabled_plugins');
      $definitions = $this->getDefinitions();
      $filterBy = $enabledPlugins ? array_flip($enabledPlugins) : $definitions;
      $definitions = array_intersect_key($definitions, $filterBy);

      $this->instances = [];
      foreach ($definitions as $pluginId => $definition) {
        $plugin = $this->createInstance($pluginId);
        assert($plugin instanceof FileVisibilityPluginInterface);
        if ($plugin->applies()) {
          $this->instances[$pluginId] = $plugin;
        }
      }
    }
    return $this->instances;
  }

  /**
   * Returns a plugin instance.
   *
   * @param string $pluginId
   *   The plugin ID.
   *
   * @return \Drupal\file_visibility\FileVisibilityPluginInterface
   *   The plugin.
   *
   * @throws \InvalidArgumentException
   *   If the plugin is not found.
   */
  public function getPlugin(string $pluginId): FileVisibilityPluginInterface {
    foreach ($this->getPlugins() as $id => $plugin) {
      if ($id === $pluginId) {
        return $plugin;
      }
    }
    throw new \InvalidArgumentException("Plugin $pluginId not found.");
  }

  /**
   * {@inheritdoc}
   */
  public function clearCachedDefinitions(): void {
    parent::clearCachedDefinitions();
    $this->instances = NULL;
  }

}
